package ru.yandex.direct.core.entity.feed.service;

import java.util.List;
import java.util.Map;
import java.util.Set;

import ru.yandex.direct.core.entity.feed.UploadedToMdsFeedInformation;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.model.MasterSystem;
import ru.yandex.direct.core.entity.feed.model.UpdateStatus;
import ru.yandex.direct.core.entity.feed.repository.FeedRepository;
import ru.yandex.direct.core.entity.feed.validation.AddFeedValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.gemini.GeminiClient;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.add.ModelsValidatedStep;
import ru.yandex.direct.operation.add.SimpleAbstractAddOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.feed.FeedUtilsKt.DEFAULT_REFRESH_INTERVAL;
import static ru.yandex.direct.core.entity.feed.FeedUtilsKt.normalizeSiteUrl;
import static ru.yandex.direct.core.entity.feed.model.MasterSystem.DIRECT;
import static ru.yandex.direct.libs.mirrortools.utils.HostingsHandler.stripDomainTail;
import static ru.yandex.direct.libs.mirrortools.utils.HostingsHandler.stripProtocol;
import static ru.yandex.direct.utils.CommonUtils.nvl;

public class FeedAddOperation extends SimpleAbstractAddOperation<Feed, Long> {

    private static final UpdateStatus DEFAULT_UPDATE_STATUS = UpdateStatus.NEW;
    private static final MasterSystem DEFAULT_MASTER_SYSTEM = DIRECT;

    private final FeedRepository feedRepository;
    private final FeedUploaderService feedUploaderService;
    private final AddFeedValidationService addFeedValidationService;
    private final GeminiClient geminiClient;
    private final boolean isSiteFeedsAllowed;

    private final int shard;
    private final ClientId clientId;
    private final Long chiefUid;
    private final Long operatorUid;

    public FeedAddOperation(
            int shard,
            ClientId clientId,
            Long chiefUid,
            Long operatorUid,
            List<Feed> feeds,
            boolean isSiteFeedsAllowed,
            AddFeedValidationService addFeedValidationService,
            FeedRepository feedRepository,
            FeedUploaderService feedUploaderService,
            GeminiClient geminiClient) {
        super(Applicability.PARTIAL, feeds);
        this.addFeedValidationService = addFeedValidationService;
        this.shard = shard;
        this.clientId = clientId;
        this.chiefUid = chiefUid;
        this.operatorUid = operatorUid;
        this.isSiteFeedsAllowed = isSiteFeedsAllowed;
        this.feedRepository = feedRepository;
        this.feedUploaderService = feedUploaderService;
        this.geminiClient = geminiClient;
    }

    @Override
    protected void validate(ValidationResult<List<Feed>, Defect> preValidationResult) {
        ValidationResult<List<Feed>, Defect> validationResult = addFeedValidationService.validate(shard, clientId,
                chiefUid, operatorUid, getModels(), isSiteFeedsAllowed);
        preValidationResult.merge(validationResult);
    }

    @Override
    protected void onModelsValidated(ModelsValidatedStep<Feed> modelsValidatedStep) {
        modelsValidatedStep.getValidModelsMap().values().forEach(s -> s.setClientId(clientId.asLong()));
    }

    @Override
    protected void beforeExecution(Map<Integer, Feed> validModelsMapToApply) {
        validModelsMapToApply.forEach((id, f) -> {
            if (f.getUpdateStatus() == null) {
                f.setUpdateStatus(DEFAULT_UPDATE_STATUS);
            }
            if (f.getMasterSystem() == null) {
                f.setMasterSystem(DEFAULT_MASTER_SYSTEM);
            }
            switch (f.getSource()) {
                case FILE:
                    prepareFileFeed(f);
                    break;
                case SITE:
                    prepareSiteFeed(f);
                    break;
                case URL:
                    prepareUrlFeed(f);
                    break;
                default:
                    throw new IllegalStateException();
            }
        });
    }

    private void prepareFileFeed(Feed f) {
        f.setRefreshInterval(0L);
        byte[] fileData = f.getFileData();
        UploadedToMdsFeedInformation uploadedFeedInfo = feedUploaderService.uploadToMds(clientId, fileData);
        f.withUrl(uploadedFeedInfo.getUrl());
        f.withCachedFileHash(uploadedFeedInfo.getFileHash());
    }

    private void prepareSiteFeed(Feed f) {
        if (f.getRefreshInterval() == null) {
            f.setRefreshInterval(DEFAULT_REFRESH_INTERVAL);
        }
        String siteUrl = normalizeSiteUrl(f.getUrl());
        f.setUrl(siteUrl);
        String mainMirror = geminiClient.getMainMirrors(Set.of(siteUrl)).get(siteUrl);
        String targetDomain = stripDomainTail(stripProtocol(nvl(mainMirror, siteUrl))).trim();
        f.setTargetDomain(targetDomain);
    }

    private void prepareUrlFeed(Feed f) {
        if (f.getRefreshInterval() == null) {
            f.setRefreshInterval(DEFAULT_REFRESH_INTERVAL);
        }
    }

    @Override
    protected List<Long> execute(List<Feed> feeds) {
        return feedRepository.add(shard, feeds);
    }

}
