package ru.yandex.direct.core.entity.feed.service;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.log.service.CommonDataLogService;
import ru.yandex.direct.core.entity.feed.model.FeedSimple;
import ru.yandex.direct.core.entity.feed.repository.FeedRepository;
import ru.yandex.direct.core.entity.feed.validation.DeleteFeedValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.market.client.exception.MarketClientException;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.operationwithid.AbstractOperationWithId;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.feed.FeedUtilsKt.logFeedStatusDelete;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

public class FeedDeleteOperation extends AbstractOperationWithId {
    private static final Logger logger = LoggerFactory.getLogger(FeedDeleteOperation.class);

    private final DeleteFeedValidationService deleteFeedValidationService;
    private final FeedRepository feedRepository;
    private final RbacService rbacService;
    private final MbiService mbiService;
    private final CommonDataLogService commonDataLogService;

    private final int shard;
    private final ClientId clientId;
    private final Long clientUid;
    private final Long operatorUid;

    private List<FeedSimple> dataCampFeeds;

    FeedDeleteOperation(
            int shard,
            ClientId clientId,
            Long clientUid,
            Long operatorUid,
            List<Long> feedIds,
            DeleteFeedValidationService deleteFeedValidationService,
            FeedRepository feedRepository,
            RbacService rbacService,
            MbiService mbiService,
            CommonDataLogService commonDataLogService
    ) {
        super(Applicability.PARTIAL, feedIds);
        this.shard = shard;
        this.clientId = clientId;
        this.clientUid = clientUid;
        this.operatorUid = operatorUid;
        this.deleteFeedValidationService = deleteFeedValidationService;
        this.feedRepository = feedRepository;
        this.rbacService = rbacService;
        this.mbiService = mbiService;
        this.commonDataLogService = commonDataLogService;
    }

    @Override
    protected ValidationResult<List<Long>, Defect> validate(List<Long> feedIds) {
        return deleteFeedValidationService.validate(shard, clientId, clientUid, operatorUid, feedIds);
    }

    @Override
    protected void beforeExecution(List<Long> feedIds) {
        this.dataCampFeeds = filterList(feedRepository.getSimple(shard, feedIds),
                feed -> feed.getMarketShopId() != null);
    }

    @Override
    protected void execute(List<Long> feedIds) {
        if (feedIds.isEmpty()) {
            return;
        }
        feedRepository.delete(shard, feedIds);
    }

    @Override
    protected void afterExecution(List<Long> feedIds) {
        if (feedIds.isEmpty()) {
            return;
        }
        long chiefUid = rbacService.getChiefByClientId(clientId);
        dataCampFeeds.forEach(feed -> {
            try {
                mbiService.sendFeed(clientId, chiefUid, feed, false);
            } catch (MarketClientException e) {
                logger.warn("Failed to disable shop '{}' after deletion of feed '{}'", feed.getMarketShopId(), feed.getId(), e);
            }
        });
        logFeedStatusDelete(shard, feedIds, commonDataLogService);
    }
}
