package ru.yandex.direct.core.entity.feed.service

import org.jooq.DSLContext
import org.springframework.stereotype.Service
import ru.yandex.direct.bmapi.client.model.BmApiErrorCode
import ru.yandex.direct.bmapi.client.model.BmApiErrorCode.Companion.FEED_TYPE_ERROR_MESSAGE
import ru.yandex.direct.bmapi.client.model.BmApiFeedInfoResponse
import ru.yandex.direct.common.TranslationService
import ru.yandex.direct.common.log.service.CommonDataLogService
import ru.yandex.direct.core.entity.adgroup.model.AdGroupWithFeedId
import ru.yandex.direct.core.entity.adgroup.model.PerformanceAdGroup
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository
import ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository
import ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetConstants.ALLOWED_DYNAMIC_AD_GROUP_TYPES
import ru.yandex.direct.core.entity.feature.service.FeatureService
import ru.yandex.direct.core.entity.feed.FATAL_BL_ERRORS
import ru.yandex.direct.core.entity.feed.FEED_FEATURE_TO_FEED_TYPE_MAP
import ru.yandex.direct.core.entity.feed.FeedTranslations
import ru.yandex.direct.core.entity.feed.bmApiFeedInfoResponseHasErrors
import ru.yandex.direct.core.entity.feed.container.FeedQueryFilter
import ru.yandex.direct.core.entity.feed.converter.FeedConverter.convertToManualFeed
import ru.yandex.direct.core.entity.feed.converter.FeedConverter.convertToManualFeedChanges
import ru.yandex.direct.core.entity.feed.getFeedsAppliedChangesForErrorBLResponse
import ru.yandex.direct.core.entity.feed.getFeedsAppliedChangesForSuccessfulBLResponse
import ru.yandex.direct.core.entity.feed.getOffersCountForCategoryInFeed
import ru.yandex.direct.core.entity.feed.logFeedStatusUpdate
import ru.yandex.direct.core.entity.feed.model.Feed
import ru.yandex.direct.core.entity.feed.model.FeedCategory
import ru.yandex.direct.core.entity.feed.model.FeedHistoryItem
import ru.yandex.direct.core.entity.feed.model.FeedHistoryItemParseResults
import ru.yandex.direct.core.entity.feed.model.FeedHistoryItemParseResultsDefect
import ru.yandex.direct.core.entity.feed.model.FeedSimple
import ru.yandex.direct.core.entity.feed.model.Source
import ru.yandex.direct.core.entity.feed.model.UpdateStatus
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import ru.yandex.direct.core.entity.feed.repository.FeedSupplementaryDataRepository
import ru.yandex.direct.core.entity.feed.toDbCategoryId
import ru.yandex.direct.core.entity.feed.toFeedCategory
import ru.yandex.direct.core.entity.feed.toFeedHistoryItem
import ru.yandex.direct.core.entity.feed.toFeedTypes
import ru.yandex.direct.core.entity.feed.validateFeedIdsToUpdate
import ru.yandex.direct.core.entity.feed.validation.AddFeedValidationService
import ru.yandex.direct.core.entity.feed.validation.DeleteFeedValidationService
import ru.yandex.direct.core.entity.feed.validation.FeedDefects
import ru.yandex.direct.core.entity.feed.validation.UpdateFeedValidationService
import ru.yandex.direct.core.entity.feedoffer.model.FeedOffer
import ru.yandex.direct.core.entity.image.container.ImageFilterContainer
import ru.yandex.direct.core.entity.image.repository.ImageDataRepository
import ru.yandex.direct.core.entity.performancefilter.repository.PerformanceFilterRepository
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.gemini.GeminiClient
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.rbac.RbacService
import ru.yandex.direct.result.MassResult
import ru.yandex.direct.result.Result
import ru.yandex.direct.validation.result.ValidationResult
import java.math.BigInteger
import java.time.LocalDateTime
import java.time.temporal.ChronoUnit
import java.util.EnumSet
import javax.annotation.Nullable

private const val MAX_VENDORS_PER_FEED = 1000
private val AVATARS_URL_REGEX = """https?://avatars.mdst?.yandex.net/get-\w+/\w+/(\w+)/\w+""".toRegex()
private val ADGROUP_TYPES_WITH_FEEDS =
    EnumSet.of(PhrasesAdgroupType.dynamic, PhrasesAdgroupType.performance, PhrasesAdgroupType.base)
const val REFRESH_HOURS_PERIOD = 1

@Service
class FeedService(
    private val shardHelper: ShardHelper,
    private val feedRepository: FeedRepository,
    private val feedSupplementaryDataRepository: FeedSupplementaryDataRepository,
    private val adGroupRepository: AdGroupRepository,
    private val bannerCommonRepository: BannerCommonRepository,
    private val performanceFilterRepository: PerformanceFilterRepository,
    private val ppcDslContextProvider: DslContextProvider,
    private val addFeedValidationService: AddFeedValidationService,
    private val updateFeedValidationService: UpdateFeedValidationService,
    private val deleteFeedValidationService: DeleteFeedValidationService,
    private val feedUploaderService: FeedUploaderService,
    private val featureService: FeatureService,
    private val rbacService: RbacService,
    private val mbiService: MbiService,
    private val commonDataLogService: CommonDataLogService,
    private val translationService: TranslationService,
    private val geminiClient: GeminiClient,
    private val imageDataRepository: ImageDataRepository
) {
    fun getFeeds(clientId: ClientId, feedQueryFilter: FeedQueryFilter): List<Feed> {
        val shard = shardHelper.getShardByClientId(clientId)
        return feedRepository[shard, clientId, feedQueryFilter]
    }

    fun getFeeds(clientId: ClientId, feedIds: Collection<Long?>): List<Feed> {
        val nonNullFeedIds = feedIds.filterNotNull()
        if (nonNullFeedIds.isEmpty()) {
            return emptyList()
        }
        val shard = shardHelper.getShardByClientId(clientId)
        val filter = FeedQueryFilter.newBuilder()
            .withFeedIds(feedIds)
            .build()
        return feedRepository[shard, clientId, filter]
    }

    fun getFeedsSimple(shard: Int, feedIds: Collection<Long?>?): List<FeedSimple> {
        val nonNullFeedIds = feedIds?.filterNotNull()
        return if (nonNullFeedIds.isNullOrEmpty()) {
            emptyList()
        } else {
            feedRepository.getSimple(shard, nonNullFeedIds!!)
        }
    }

    fun getFeedsSimple(clientId: ClientId, feedQueryFilter: FeedQueryFilter): List<FeedSimple> {
        val shard = shardHelper.getShardByClientId(clientId)
        return feedRepository.getSimple(shard, clientId, feedQueryFilter)
    }

    fun getFeedCategories(clientId: ClientId, feedIds: List<Long?>): List<FeedCategory> {
        val nonNullFeedIds = feedIds.filterNotNull()
        if (nonNullFeedIds.isEmpty()) {
            return emptyList()
        }
        val shard = shardHelper.getShardByClientId(clientId)
        return feedSupplementaryDataRepository.getFeedCategories(shard, nonNullFeedIds)
    }

    fun getLatestFeedHistoryItems(clientId: ClientId, feedIds: Collection<Long>): Map<Long, FeedHistoryItem> {
        val shard = shardHelper.getShardByClientId(clientId)
        return feedSupplementaryDataRepository.getLatestFeedHistoryItems(shard, feedIds)
    }

    fun getFeedByPerformanceFilterId(clientId: ClientId, filterIds: List<Long>): Map<Long, Feed> {
        val shard = shardHelper.getShardByClientId(clientId)
        val filtersById = performanceFilterRepository.getFiltersById(shard, filterIds)
        val filterIdsByAdGroupId = filtersById.groupBy({ it.pid }) { it.id }
        val feedByAdGroupId = getFeedByPerformanceAdGroupId(clientId, filterIdsByAdGroupId.keys)
        val result: MutableMap<Long, Feed> = HashMap()
        filterIdsByAdGroupId.forEach { (adGroupId, agGroupFilterIds) ->
            agGroupFilterIds.forEach { result[it] = feedByAdGroupId[adGroupId]!! }
        }
        return result
    }

    fun getFeedByPerformanceAdGroupId(clientId: ClientId, adGroupIds: Collection<Long>): Map<Long, Feed> {
        val shard = shardHelper.getShardByClientIdStrictly(clientId)
        val adGroups = adGroupRepository.getAdGroups(shard, adGroupIds)
        val feedIdByAdGroupId = adGroups.asSequence()
            .filterIsInstance<PerformanceAdGroup>()
            .associateBy({ it.id }) { it.feedId }
        return getFeedByAdGroupId(shard, clientId, feedIdByAdGroupId)
    }

    fun getFeedByDynamicAdGroupId(clientId: ClientId, adGroupIds: Collection<Long>): Map<Long, Feed> {
        val shard = shardHelper.getShardByClientIdStrictly(clientId)
        val adGroups = adGroupRepository.getAdGroups(shard, adGroupIds)
        val feedIdByAdGroupId = adGroups.asSequence()
            .filter { ALLOWED_DYNAMIC_AD_GROUP_TYPES.contains(it.type) }
            .filterIsInstance<AdGroupWithFeedId>()
            .associateBy({ it.id }) { it.feedId }
        return getFeedByAdGroupId(shard, clientId, feedIdByAdGroupId)
    }

    private fun getFeedByAdGroupId(
        shard: Int,
        clientId: ClientId,
        feedIdByAdGroupId: Map<Long, Long>
    ): Map<Long, Feed> {
        val feedIds = feedIdByAdGroupId.values.toSet()
        val feeds = feedRepository.get(shard, clientId, feedIds)
        val feedByFeedId = feeds.associateBy { it.id }
        return feedIdByAdGroupId
            .mapValues { feedByFeedId[it.value] }
            .filterValues { it != null }
            .toMap() as Map<Long, Feed>
    }

    fun getFeedsWithVendors(clientId: ClientId, feedQueryFilter: FeedQueryFilter): List<Feed> {
        val shard = shardHelper.getShardByClientId(clientId)
        val feeds = feedRepository.get(shard, clientId, feedQueryFilter)
        val feedIds = feeds.map { it.id }
        val feedVendors = feedSupplementaryDataRepository.getFeedVendorsByFeedId(shard, feedIds)
        feeds.forEach { feed ->
            val vendors = feedVendors[feed.id]?.map { it.name }
            feed.topVendors = vendors ?: emptyList()
        }
        return feeds
    }

    /**
     * Возвращает фид, созданный по бизнесу маркетплейса (`feeds.master_system == 'shop_in_shop'`).
     */
    @Nullable
    fun getShopInShopFeedByUrl(clientId: ClientId, url: String): Feed? {
        val shard = shardHelper.getShardByClientId(clientId)
        return feedRepository.getShopInShopFeedByUrl(shard, clientId, url)
    }

    /**
     * Массовое добавление фидов для клиента.
     * Считается успешным, если добавлен хотя бы один фид.
     *
     * @param clientId    идентификатор клиента
     * @param chiefUid    uid главного представителя клиента
     * @param operatorUid uid оператора
     * @param feeds       добавлемые идентификаторы фидов
     * @return результат добавления фидов
     */
    fun addFeeds(
        clientId: ClientId,
        chiefUid: Long,
        operatorUid: Long,
        feeds: List<Feed>
    ): MassResult<Long> {
        val shard = shardHelper.getShardByClientIdStrictly(clientId)
        val isSiteFeedsAllowed = featureService.isEnabledForClientId(clientId, FeatureName.SITE_FEEDS_ALLOWED)
        val feedAddOperation = FeedAddOperation(
            shard,
            clientId,
            chiefUid,
            operatorUid,
            feeds,
            isSiteFeedsAllowed,
            addFeedValidationService,
            feedRepository,
            feedUploaderService,
            geminiClient
        )
        return feedAddOperation.prepareAndApply()
    }

    /**
     * Массовое обновление фидов для клиента.
     * Считается успешным, если обновлен хотя бы один фид.
     *
     * @param clientId    идентификатор клиента
     * @param chiefUid    uid главного представителя клиента
     * @param operatorUid uid оператора
     * @param feeds       добавлемые идентификаторы фидов
     * @return результат добавления фидов
     */
    fun updateFeeds(
        clientId: ClientId,
        chiefUid: Long,
        operatorUid: Long,
        feeds: List<ModelChanges<Feed>>
    ): MassResult<Long> {
        val shard = shardHelper.getShardByClientIdStrictly(clientId)
        val feedUpdateOperation = FeedUpdateOperation(
            shard,
            clientId,
            chiefUid,
            operatorUid,
            feeds,
            adGroupRepository,
            bannerCommonRepository,
            feedRepository,
            updateFeedValidationService,
            feedUploaderService,
            commonDataLogService,
            geminiClient
        )
        return feedUpdateOperation.prepareAndApply()
    }

    /**
     * Массовое удаление фидов для клиента.
     * Считается успешным, если удален хотя бы один фид.
     *
     * @param clientId    идентификатор клиента
     * @param clientUid   uid клиента
     * @param operatorUid uid оператора
     * @param feedIds     удаляемые идентификаторы фидов
     * @return результат удаления фидов
     */
    fun deleteFeeds(
        clientId: ClientId,
        clientUid: Long,
        operatorUid: Long,
        feedIds: List<Long>
    ): MassResult<Long> {
        val shard = shardHelper.getShardByClientIdStrictly(clientId)
        val feedDeleteOperation = FeedDeleteOperation(
            shard,
            clientId,
            clientUid,
            operatorUid,
            feedIds,
            deleteFeedValidationService,
            feedRepository,
            rbacService,
            mbiService,
            commonDataLogService
        )
        return feedDeleteOperation.prepareAndApply()
    }

    fun getFeedsToSendToMBI(shard: Int): List<FeedSimple> {
        return feedRepository.getFeedsToSendToMBI(shard)
            .groupBy { ClientId.fromLong(it.clientId) }
            .flatMap { (clientId, feeds) ->
                val allowedToSendFeedTypesFeatureStates = featureService.getFeatureStates(
                    clientId,
                    FEED_FEATURE_TO_FEED_TYPE_MAP.keys
                )
                val feedTypes = toFeedTypes(allowedToSendFeedTypesFeatureStates)
                val sendUnused = featureService.isEnabledForClientId(clientId, FeatureName.SEND_UNUSED_FEEDS_TO_MBI)
                feeds.asSequence()
                    .filter { it.source == Source.SITE || it.feedType in feedTypes }
                    .filter { it.source == Source.SITE || sendUnused || !it.usageTypes.isNullOrEmpty() }
            }
    }

    /**
     * Метод возвращает фид, созданный по вручную добавленным офферам (`feeds.master_system == 'manual'`).
     * Таких фидов не может быть больше одного для клиента
     */
    @Nullable
    fun getManuallyAddedFeed(clientId: ClientId): FeedSimple? {
        val shard = shardHelper.getShardByClientId(clientId)
        return feedRepository.getManuallyAddedFeed(shard, clientId)
    }

    fun addOrUpdateManualFeed(
        clientId: ClientId,
        subjectUserUid: Long,
        operatorUid: Long,
        offers: List<FeedOffer>
    ): MassResult<Long> {
        val imageUrlToHash = offers.asSequence()
            .mapNotNull {
                val imageUrl = it.images[0]
                val imageHash = AVATARS_URL_REGEX.matchEntire(imageUrl)?.groupValues?.get(1)
                if (imageHash != null) imageUrl to imageHash else null
            }
            .take(10)
            .toMap()
        val imageHashToSize = imageDataRepository.getImages(
            shardHelper.getShardByClientId(clientId),
            clientId,
            ImageFilterContainer().withImageHashes(imageUrlToHash.values.toSet())
        ).associate { it.imageHash to it.size }

        val imageUrlToSize = imageUrlToHash.mapValues { imageHashToSize[it.value] }

        val feed = getManuallyAddedFeed(clientId)
        return if (feed == null) {
            val name = translationService.translate(FeedTranslations.INSTANCE.manualFeedName())
            val feedToAdd = convertToManualFeed(name, offers, imageUrlToSize)
            addFeeds(clientId, subjectUserUid, operatorUid, listOf(feedToAdd))
        } else {
            val feedToUpdate = convertToManualFeedChanges(feed.id, offers, imageUrlToSize)
            updateFeeds(clientId, subjectUserUid, operatorUid, listOf(feedToUpdate))
        }
    }

    fun getFeedIdsByAdGroupIds(shard: Int, adGroupIds: Collection<Long>?): Map<Long, Long> {
        return if (adGroupIds.isNullOrEmpty()) {
            emptyMap()
        } else feedRepository.getFeedIdsByAdGroupIds(shard, adGroupIds)
            .filterValues { it != null }
    }

    fun getFeedIdsByCampaignIds(shard: Int, campaignIds: Set<Long>?): Map<Long, Set<Long>> {
        if (campaignIds.isNullOrEmpty()) {
            return emptyMap()
        }
        val adGroupIdsByCampaignIds =
            adGroupRepository.getAdGroupIdsByCampaignIdsWithTypes(shard, campaignIds, ADGROUP_TYPES_WITH_FEEDS)

        val adGroupIds = adGroupIdsByCampaignIds.values.asSequence()
            .flatten()
            .filterNotNull()
            .toSet()
        val feedIdsByAdGroupIds = getFeedIdsByAdGroupIds(shard, adGroupIds)
        return adGroupIdsByCampaignIds
            .mapValues { (_, groupIds) ->
                groupIds.mapNotNullTo(HashSet()) { feedIdsByAdGroupIds[it] }
            }.filterValues { it.isNotEmpty() }
    }

    fun saveFeedFromBmApiResponse(shard: Int, feedIdToBlFeedInfo: Map<Long, BmApiFeedInfoResponse>) {
        val feedsToUpdate = feedRepository.getSimpleUpdatingFeeds(shard, feedIdToBlFeedInfo.keys)

        val (validFeedIdsToUpdate, feedIdsWithIncorrectFeedType) = validateFeedIdsToUpdate(
            feedsToUpdate,
            feedIdToBlFeedInfo
        )
        val validFeedIdToBlFeedInfo = feedIdToBlFeedInfo.filterKeys(validFeedIdsToUpdate::contains)
        val existingFeedCategoriesByFeedId =
            feedSupplementaryDataRepository.getFeedCategoriesByFeedId(shard, validFeedIdsToUpdate)

        val successResponses = validFeedIdToBlFeedInfo
            .filterValues { bmApiFeedInfoResponse -> !bmApiFeedInfoResponseHasErrors(bmApiFeedInfoResponse) }
        val errorResponses = feedIdToBlFeedInfo
            .filterValues { bmApiFeedInfoResponse -> bmApiFeedInfoResponseHasErrors(bmApiFeedInfoResponse) }

        val now = LocalDateTime.now()
        val feedsSuccessAppliedChanges =
            getFeedsAppliedChangesForSuccessfulBLResponse(feedsToUpdate, successResponses, now)
        val feedsErrorAppliedChanges =
            getFeedsAppliedChangesForErrorBLResponse(feedsToUpdate, errorResponses, feedIdsWithIncorrectFeedType, now)
        val allFeedAppliedChanges = feedsSuccessAppliedChanges + feedsErrorAppliedChanges

        ppcDslContextProvider.ppcTransaction(shard) { config ->
            val context = config.dsl()
            feedRepository.update(context, allFeedAppliedChanges)
            updateCategories(context, validFeedIdsToUpdate, existingFeedCategoriesByFeedId, validFeedIdToBlFeedInfo)
            updateVendors(context, validFeedIdsToUpdate, validFeedIdToBlFeedInfo)
            updateHistory(context, errorResponses, successResponses, feedIdsWithIncorrectFeedType, now)
        }
        logFeedStatusUpdate(shard, allFeedAppliedChanges, commonDataLogService)
    }

    private fun updateHistory(
        dslContext: DSLContext,
        errorResponses: Map<Long, BmApiFeedInfoResponse>,
        successResponses: Map<Long, BmApiFeedInfoResponse>,
        feedIdsWithIncorrectFeedType: List<Long>,
        now: LocalDateTime
    ) {
        val hasFatalOrUnknownErrors = errorResponses.filterValues { errorResponse ->
            errorResponse.errors!!.any { error -> FATAL_BL_ERRORS.contains(error.error)  }
        }
        val historyItemsToInsert: MutableList<FeedHistoryItem> = mutableListOf()
        historyItemsToInsert.addAll(errorResponses
            .asSequence()
            .filter { (feedId) -> !hasFatalOrUnknownErrors.containsKey(feedId) }
            .map { (feedId, response) ->
                toFeedHistoryItem(feedId, now, response)
            })

        historyItemsToInsert.addAll(feedIdsWithIncorrectFeedType.map { feedId ->
            FeedHistoryItem().withFeedId(feedId)
                .withCreatedAt(now)
                .withOfferCount(0)
                .withParseResults(
                    FeedHistoryItemParseResults().withErrors(
                        listOf(
                            FeedHistoryItemParseResultsDefect()
                                .withMessageEn(FEED_TYPE_ERROR_MESSAGE)
                                .withCode(BmApiErrorCode.BL_ERROR_FEED_TYPE_MISMATCH.code.toLong())
                        )
                    )
                )
        })

        historyItemsToInsert.addAll(successResponses.map { (feedId, response) ->
            toFeedHistoryItem(feedId, now, response)
        })

        feedSupplementaryDataRepository.addHistoryItems(dslContext, historyItemsToInsert)
    }

    private fun updateCategories(
        dslContext: DSLContext,
        feedIds: List<Long>,
        existingCategoriesByFeedId: Map<Long, List<FeedCategory>>,
        feedIdToBlFeedInfo: Map<Long, BmApiFeedInfoResponse>
    ) {
        val categoriesFromBmApiByFeedId =
            feedIdToBlFeedInfo.mapValues { entry -> entry.value.categories }
                .filterValues { it != null }

        val categoriesToDelete: MutableList<FeedCategory> = mutableListOf()
        val categoriesToUpdate: MutableList<AppliedChanges<FeedCategory>> = mutableListOf()
        val categoriesToAdd: MutableList<FeedCategory> = mutableListOf()

        feedIds.forEach { feedId ->
            val new = categoriesFromBmApiByFeedId[feedId]?.associate { BigInteger(it.id) to it }
            val old = existingCategoriesByFeedId[feedId]?.associate { it.categoryId to it }
            if (old == null && new == null) {
                return@forEach
            }

            if (old == null && new != null) {
                categoriesToAdd.addAll(new.values.map { category ->
                    toFeedCategory(feedId, category, feedIdToBlFeedInfo[feedId]!!)
                })
                return@forEach
            }
            if (old != null && new == null) {
                categoriesToDelete.addAll(old.values)
                return@forEach
            }

            categoriesToDelete.addAll(old!!.filterNot { entry -> new!!.keys.contains(entry.key) }.values)

            new!!.filterNot { entry -> old.keys.contains(entry.key) }.forEach { (_, category) ->
                categoriesToAdd.add(
                    toFeedCategory(feedId, category, feedIdToBlFeedInfo[feedId]!!)
                )
            }

            new.filter { entry -> old.keys.contains(entry.key) }.forEach { (categoryId, category) ->
                val mc = ModelChanges(old[categoryId]!!.id, FeedCategory::class.java)
                mc.process(category.name, FeedCategory.NAME)
                mc.process(toDbCategoryId(category.parentId), FeedCategory.PARENT_CATEGORY_ID)
                mc.process(
                    getOffersCountForCategoryInFeed(feedIdToBlFeedInfo[feedId]!!, category),
                    FeedCategory.OFFER_COUNT
                )
                mc.process(false, FeedCategory.IS_DELETED)
                categoriesToUpdate.add(mc.applyTo(old[categoryId]!!))
            }
        }

        feedSupplementaryDataRepository.markFeedCategoriesDeleted(dslContext, categoriesToDelete)
        // пара (feedId, categoryId) в БД -- уникальный ключ, ноль мы проставляем
        feedSupplementaryDataRepository.addCategories(
            dslContext,
            categoriesToAdd.filter { it.categoryId != BigInteger.ZERO })
        feedSupplementaryDataRepository.updateCategories(
            dslContext,
            categoriesToUpdate.filter {
                it.model.categoryId != BigInteger.ZERO && it.getNewValue(FeedCategory.CATEGORY_ID) != BigInteger.ZERO
            }
        )
    }

    private fun updateVendors(
        dslContext: DSLContext,
        feedIdsToUpdate: List<Long>,
        feedIdToBlFeedInfo: Map<Long, BmApiFeedInfoResponse>
    ) {
        val newTopVendorsByFeedId = feedIdToBlFeedInfo.filterValues { s -> s.vendorsToOffersCount != null }
            .mapValues { (_, blFeedInfo) ->
                blFeedInfo.vendorsToOffersCount!!.toList().sortedBy { (_, value) -> value }
                    .subList(0, minOf(MAX_VENDORS_PER_FEED - 1, blFeedInfo.vendorsToOffersCount!!.size))
                    .map { it.first }
            }
        feedSupplementaryDataRepository.deleteFeedVendors(dslContext, feedIdsToUpdate)
        feedSupplementaryDataRepository.addFeedVendors(dslContext, newTopVendorsByFeedId)
    }

    fun refreshFeed(clientId: ClientId, chiefUid: Long, operatorUid: Long, feedId: Long): Result<Long> {
        val shard = shardHelper.getShardByClientId(clientId)
        val feeds = feedRepository.get(shard, setOf(feedId))
        if (feeds.isEmpty()) {
            val validationResult = ValidationResult.failed(feedId, AdGroupDefects.feedNotExist(feedId))
            return Result.broken(validationResult)
        }
        val feed = feeds[0]
        if (feed.source != Source.URL) {
            val validationResult =
                ValidationResult.failed(feedId, FeedDefects.feedWithoutUrlSourceCannotBeRefreshed())
            return Result.broken(validationResult)
        }
        if (feed.updateStatus == UpdateStatus.NEW) {
            val validationResult =
                ValidationResult.failed(feedId, FeedDefects.feedWithStatusNewCannotBeRefreshed())
            return Result.broken(validationResult)
        }
        val now = LocalDateTime.now()
        val hours = ChronoUnit.HOURS.between(feed.lastChange, now)
        if (hours < REFRESH_HOURS_PERIOD) {
            val validationResult = ValidationResult.failed(
                feedId,
                FeedDefects.feedCannotBeRefreshedMoreOftenThanRefreshHoursPeriod(REFRESH_HOURS_PERIOD.toLong())
            )
            return Result.broken(validationResult)
        }
        val modelChanges = ModelChanges(
            feed.id,
            Feed::class.java
        ).process(UpdateStatus.NEW, Feed.UPDATE_STATUS)

        val result = updateFeeds(clientId, chiefUid, operatorUid, listOf(modelChanges))
        // Ожидается только ошибка уровня операции NO_RIGHTS, других ошибок валидации не ожидается
        return if (result.errors.isEmpty()) {
            Result.successful(feedId, ValidationResult.success(feed.id))
        } else {
            Result.broken(ValidationResult.failed(feedId, result.errors[0].defect))
        }
    }

    /**
     * Метод достает из шарда iterationCapacity фидов в статусе New,
     * со старейшей датой feed.LastChange, для отправки в BmAPI
     * Фиды с source=SITE должны быть зарегистрированы в MBI до отправки в BmAPI
     *
     * @param shard шард из которого доставать фиды
     * @param iterationCapacity сколько фидов доставать
     * @return список упрощенных представлений фидов
     */
    fun getNewFeedsToSendToBmAPI(shard: Int, iterationCapacity: Int): List<FeedSimple> {
        return feedRepository.getNewFeedsToSendToBmAPI(shard, iterationCapacity)
    }
}
