package ru.yandex.direct.core.entity.feed.service;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import ru.yandex.direct.common.log.service.CommonDataLogService;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.feed.UploadedToMdsFeedInformation;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.model.MasterSystem;
import ru.yandex.direct.core.entity.feed.model.StatusMBISynced;
import ru.yandex.direct.core.entity.feed.model.UpdateStatus;
import ru.yandex.direct.core.entity.feed.repository.FeedRepository;
import ru.yandex.direct.core.entity.feed.validation.UpdateFeedValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.gemini.GeminiClient;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.feed.FeedUtilsKt.logFeedStatusUpdate;
import static ru.yandex.direct.core.entity.feed.FeedUtilsKt.normalizeSiteUrl;
import static ru.yandex.direct.core.entity.feed.FeedUtilsKt.unFakeUrlIfNeeded;
import static ru.yandex.direct.core.entity.feed.model.Source.SITE;
import static ru.yandex.direct.libs.mirrortools.utils.HostingsHandler.stripDomainTail;
import static ru.yandex.direct.libs.mirrortools.utils.HostingsHandler.stripProtocol;
import static ru.yandex.direct.utils.CollectionUtils.flatToList;
import static ru.yandex.direct.utils.CommonUtils.notEquals;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class FeedUpdateOperation extends SimpleAbstractUpdateOperation<Feed, Long> {

    private final AdGroupRepository adGroupRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final FeedRepository feedRepository;
    private final UpdateFeedValidationService updateFeedValidationService;
    private final FeedUploaderService feedUploaderService;
    private final CommonDataLogService commonDataLogService;
    private final GeminiClient geminiClient;
    private final int shard;
    private final ClientId clientId;
    private final Long chiefUid;
    private final Long operatorUid;

    FeedUpdateOperation(
            int shard,
            ClientId clientId,
            Long chiefUid,
            Long operatorUid,
            List<ModelChanges<Feed>> modelChanges,
            AdGroupRepository adGroupRepository,
            BannerCommonRepository bannerCommonRepository,
            FeedRepository feedRepository,
            UpdateFeedValidationService updateFeedValidationService,
            FeedUploaderService feedUploaderService,
            CommonDataLogService commonDataLogService,
            GeminiClient geminiClient) {
        super(Applicability.PARTIAL, modelChanges, id -> new Feed().withId(id));
        this.shard = shard;
        this.clientId = clientId;
        this.chiefUid = chiefUid;
        this.operatorUid = operatorUid;
        this.adGroupRepository = adGroupRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.feedRepository = feedRepository;
        this.updateFeedValidationService = updateFeedValidationService;
        this.feedUploaderService = feedUploaderService;
        this.commonDataLogService = commonDataLogService;
        this.geminiClient = geminiClient;
    }

    @Override
    protected ValidationResult<List<ModelChanges<Feed>>, Defect> validateModelChanges(
            List<ModelChanges<Feed>> modelChanges
    ) {
        return updateFeedValidationService.validate(shard, clientId, chiefUid, operatorUid, modelChanges);
    }

    @Override
    protected ValidationResult<List<Feed>, Defect> validateAppliedChanges(
            ValidationResult<List<Feed>, Defect> validationResult
    ) {
        return updateFeedValidationService.validateAppliedChanges(validationResult);
    }

    @Override
    protected void beforeExecution(ExecutionStep<Feed> executionStep) {
        Collection<AppliedChanges<Feed>> validModelsMapToApply = executionStep.getAppliedChangesForExecution();
        var now = LocalDateTime.now();
        validModelsMapToApply.forEach(feedAppliedChanges -> {
            if (isFeedUrlChanged(feedAppliedChanges)
                    || feedAppliedChanges.changed(Feed.LOGIN)
                    || feedAppliedChanges.changed(Feed.PLAIN_PASSWORD)
                    || feedAppliedChanges.changed(Feed.IS_REMOVE_UTM)
            ) {
                feedAppliedChanges.modify(Feed.UPDATE_STATUS, UpdateStatus.NEW);
            }
            if (feedAppliedChanges.changed(Feed.FILE_DATA)) {
                prepareFileFeed(feedAppliedChanges);
            } else if (isFeedUrlChanged(feedAppliedChanges) && feedAppliedChanges.getModel().getSource() == SITE) {
                prepareSiteFeed(feedAppliedChanges);
            }
            //Если в Директовом фиде поменяли url, или залили новый файл(фактически тоже поменяли url),
            //или поменяли логин/пароль фида, то надо перерегистрировать фид в MBI
            if ((isFeedUrlChanged(feedAppliedChanges)
                    || feedAppliedChanges.changed(Feed.LOGIN)
                    || feedAppliedChanges.changed(Feed.PLAIN_PASSWORD))
                    && feedAppliedChanges.getModel().getMasterSystem() == MasterSystem.DIRECT
                    && feedAppliedChanges.getModel().getMarketShopId() != null) {
                feedAppliedChanges.modify(Feed.STATUS_MBI_SYNCED, StatusMBISynced.NO);
            }

            if (!isFeedUrlChanged(feedAppliedChanges)) {
                feedAppliedChanges.modify(Feed.URL, feedAppliedChanges.getOldValue(Feed.URL));
            }
            if (feedAppliedChanges.hasActuallyChangedProps()) {
                feedAppliedChanges.modify(Feed.LAST_CHANGE, now);
            }
        });
    }

    @Override
    protected Collection<Feed> getModels(Collection<Long> feedIds) {
        return feedRepository.get(shard, clientId, feedIds);
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<Feed>> applicableAppliedChanges) {
        feedRepository.update(shard, applicableAppliedChanges);
        List<Long> feedIds = mapList(applicableAppliedChanges, c -> c.getModel().getId());
        updateStatusBsSyncForBanners(feedIds);
        logFeedStatusUpdate(shard, applicableAppliedChanges, commonDataLogService);
        return feedIds;
    }

    private void updateStatusBsSyncForBanners(List<Long> feedIds) {
        Map<Long, List<Long>> adGroupIdsByFeedId = adGroupRepository.getAdGroupIdsByFeedId(shard, feedIds);
        List<Long> adGroupIds = flatToList(adGroupIdsByFeedId.values());
        bannerCommonRepository.updateStatusBsSyncedByAdgroupId(shard, adGroupIds, StatusBsSynced.NO);
    }

    private void prepareFileFeed(AppliedChanges<Feed> changes) {
        byte[] newFileData = changes.getNewValue(Feed.FILE_DATA);
        UploadedToMdsFeedInformation feedInfo = feedUploaderService.uploadToMds(clientId, newFileData);
        changes.modify(Feed.URL, feedInfo.getUrl());
        if (notEquals(changes.getModel().getCachedFileHash(), feedInfo.getFileHash())) {
            changes.modify(Feed.CACHED_FILE_HASH, feedInfo.getFileHash());
            changes.modify(Feed.UPDATE_STATUS, UpdateStatus.NEW);
        }
    }

    private void prepareSiteFeed(AppliedChanges<Feed> changes) {
        String siteUrl = normalizeSiteUrl(changes.getNewValue(Feed.URL));
        changes.modify(Feed.URL, siteUrl);
        String mainMirror = geminiClient.getMainMirrors(Set.of(siteUrl)).get(siteUrl);
        String targetDomain = stripDomainTail(stripProtocol(nvl(mainMirror, siteUrl))).trim();
        changes.modify(Feed.TARGET_DOMAIN, targetDomain);
    }

    static boolean isFeedUrlChanged(AppliedChanges<Feed> appliedChanges) {
        return appliedChanges.changed(Feed.URL)
                && (appliedChanges.getOldValue(Feed.URL) == null
                || !unFakeUrlIfNeeded(appliedChanges.getOldValue(Feed.URL))
                .equals(appliedChanges.getNewValue(Feed.URL)));
    }
}
