package ru.yandex.direct.core.entity.feed.service

import org.springframework.stereotype.Service
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesAdGroupRepository
import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesCampaignRepository
import ru.yandex.direct.core.entity.adgroup.aggrstatus.AggregatedStatusAdGroup
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum
import ru.yandex.direct.core.entity.container.CampaignIdAndAdGroupIdPair
import ru.yandex.direct.core.entity.feed.model.FeedUsageType
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import java.util.EnumSet
import java.util.Optional

@Service
class FeedUsageService constructor(
    private val feedRepository: FeedRepository,
    private val aggregatedStatusesCampaignRepository: AggregatedStatusesCampaignRepository,
    private val aggregatedStatusesAdGroupRepository: AggregatedStatusesAdGroupRepository
) {
    companion object {
        // Very similar check is in ess router RecalculateFeedUsageTypesRule. Keep it corresponding.
        private val STOP_STATUSES = setOf(
            GdSelfStatusEnum.ARCHIVED,
            GdSelfStatusEnum.STOP_CRIT,
            GdSelfStatusEnum.STOP_OK,
            GdSelfStatusEnum.STOP_WARN,
            GdSelfStatusEnum.DRAFT
        )

        fun isObjectStatusStopped(optionalSelfStatus: Optional<GdSelfStatusEnum>): Boolean =
            STOP_STATUSES.contains(optionalSelfStatus.orElse(GdSelfStatusEnum.RUN_OK))
    }

    /**
     * Метод, который для фидов с заданными feedId рассчитывает их используемость на текущий момент
     */
    fun getActualFeedUsageTypeByFeedId(
        shard: Int,
        feedIdsToRecalculateUsageTypes: List<Long>
    ): Map<Long, EnumSet<FeedUsageType>> {
        // Получаем id всех кампаний и групп, к которым привязаны фиды
        val campaignIdAdGroupIdByFeedIds = getCampaignIdAdGroupIdByFeedIds(shard, feedIdsToRecalculateUsageTypes)

        val campaignIdsByFeedId = campaignIdAdGroupIdByFeedIds
            .map { (campaignIdAdGroupIdPair, feedId) -> feedId to campaignIdAdGroupIdPair.campaignId }
            .groupBy({ it.first }, { it.second })
        val adGroupIdsByFeedId = campaignIdAdGroupIdByFeedIds
            .map { (campaignIdAdGroupIdPair, feedId) -> feedId to campaignIdAdGroupIdPair.adGroupId }
            .groupBy({ it.first }, { it.second })

        val campaignIdsToGetStatuses = campaignIdsByFeedId.values.flatten()
        val adgroupIdsToGetStatuses = adGroupIdsByFeedId.values.flatten()

        // Достаем статусы групп и кампаний
        val adGroupStatusesByAdGroupId =
            aggregatedStatusesAdGroupRepository.getAdGroupsWithStatusById(shard, adgroupIdsToGetStatuses)
        val campaignSelfStatusesByCampaignId =
            aggregatedStatusesCampaignRepository.getCampaignById(shard, campaignIdsToGetStatuses)
                .filter { it.value.aggregatedStatus != null }
                .mapValues { it.value.aggregatedStatus.status }

        // Определяем какие группы активны (с учетом активности кампаний)
        val activeAdGroupIds = calcActiveAdGroups(adGroupStatusesByAdGroupId, campaignSelfStatusesByCampaignId)

        fun feedIdIsUsedInActiveAdGroups(feedId: Long): Boolean {
            val adGroups = adGroupIdsByFeedId[feedId] ?: return false
            return adGroups.any { it in activeAdGroupIds }
        }

        // Определяем статус использования для фидов
        return feedIdsToRecalculateUsageTypes.associateWith {
            if (feedIdIsUsedInActiveAdGroups(it)) {
                EnumSet.of(FeedUsageType.GOODS_ADS)
            } else {
                EnumSet.noneOf(FeedUsageType::class.java)
            }
        }
    }

    private fun calcActiveAdGroups(
        adGroupStatusesByAdGroupId: Map<Long, AggregatedStatusAdGroup>,
        campaignSelfStatusesByCampaignId: Map<Long, Optional<GdSelfStatusEnum>>
    ): List<Long> {
        // Делаем мапу связи кампаний и групп
        val adGroupIdByCampaignId = adGroupStatusesByAdGroupId.values
            .associate { it.id to it.campaignId }

        val adGroupSelfStatusesByAdGroupId = adGroupStatusesByAdGroupId
            .filter { it.value.aggregatedStatus != null }
            .mapValues { it.value.aggregatedStatus.status }

        // Кампании не активны только по собственным статусам
        val stoppedCampaignIds = campaignSelfStatusesByCampaignId.filter {
            isObjectStatusStopped(it.value)
        }.map { it.key }

        // Группы не активны по собственным статусам и по статусам своих кампаний
        return adGroupSelfStatusesByAdGroupId.filterNot {
            isObjectStatusStopped(it.value) || stoppedCampaignIds.contains(adGroupIdByCampaignId[it.key])
        }.map { it.key }
    }

    fun getCampaignIdAdGroupIdByFeedIds(
        shard: Int,
        feedIds: Collection<Long>
    ): List<Pair<CampaignIdAndAdGroupIdPair, Long>> {
        val performance = feedRepository.getCampaignIdAdGroupIdByFeedsInPerformance(shard, feedIds)
        val dynamic = feedRepository.getCampaignIdAdGroupIdByFeedsInDynamic(shard, feedIds)
        return performance.toList() + dynamic.toList()
    }
}
