package ru.yandex.direct.core.entity.feed.service

import java.time.LocalDateTime
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import ru.yandex.direct.common.util.RelaxedWorker
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType.BASE
import ru.yandex.direct.core.entity.feed.createFakeFeedUrl
import ru.yandex.direct.core.entity.feed.getForceDc
import ru.yandex.direct.core.entity.feed.model.Feed
import ru.yandex.direct.core.entity.feed.model.FeedSimple
import ru.yandex.direct.core.entity.feed.model.FeedUsageType
import ru.yandex.direct.core.entity.feed.model.MasterSystem
import ru.yandex.direct.core.entity.feed.model.Source
import ru.yandex.direct.core.entity.feed.model.StatusMBIEnabled
import ru.yandex.direct.core.entity.feed.model.StatusMBISynced
import ru.yandex.direct.core.entity.feed.repository.FeedRepository
import ru.yandex.direct.core.entity.feed.unFakeUrlIfNeeded
import ru.yandex.direct.core.entity.uac.model.EcomDomain
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.gemini.GeminiClient
import ru.yandex.direct.gemini.GeminiClientException
import ru.yandex.direct.market.client.MarketClient
import ru.yandex.direct.market.client.MarketClient.FeedFeaturesInfo
import ru.yandex.direct.market.client.MarketClient.FileFeedInfo
import ru.yandex.direct.market.client.MarketClient.Schema
import ru.yandex.direct.market.client.MarketClient.SiteFeedInfo
import ru.yandex.direct.market.client.MarketClient.SitePreviewIds
import ru.yandex.direct.market.client.MarketClient.SitePreviewInfo
import ru.yandex.direct.market.client.MarketClient.UrlFeedInfo
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.utils.UrlUtils.isUrlInternal
import ru.yandex.direct.utils.model.UrlParts
import kotlin.time.Duration.Companion.days
import kotlin.time.toJavaDuration

/**
 * Сервис содержащий бизнес-логику хождения в ручки MBI
 * (@see <a href="https://abc.yandex-team.ru/services/mbi/">Маркет-интерфейс магазинов</a>)
 */
@Service
class MbiService @Autowired constructor(
    private val shardSupport: ShardHelper,
    private val marketClient: MarketClient,
    private val feedRepository: FeedRepository,
    private val geminiClient: GeminiClient,
) {

    companion object {
        private val logger: Logger = LoggerFactory.getLogger(MbiService::class.java)

        val FEED_KEEP_MBI_ENABLED_DURATION = 7.days.toJavaDuration()

        const val OFFERS_COUNT_FOR_PREVIEW = 10

        const val MAX_COUNT_OF_SITE_PREVIEWS_TO_DELETE = 100
    }

    data class SendFeedResult(
        val directFeedId: Long,
        val marketFeedId: Long,
        val shopId: Long,
        val businessId: Long,
        val fakeUrl: String,
        val statusMBIEnabled: StatusMBIEnabled
    )

    data class AddSitePreviewResult(val marketFeedId: Long, val shopId: Long, val businessId: Long)

    private data class UrlPartsResult(val schema: Schema, val domain: String)

    /**
     * Отправляет фид в MBI и возвращает значения, которые должны быть сохранены для обслуживаемого через
     * единое офферное хранилище фида. Актуализирует включенность фида.
     */
    fun sendFeed(clientId: ClientId, chiefUid: Long, feed: FeedSimple, forceEnableDirectPlacement: Boolean? = null): SendFeedResult {
        val enableDirectPlacement = forceEnableDirectPlacement ?: shouldBeEnabled(feed)
        var answer: MarketClient.SendFeedResult
        val fakeUrl: String
        // В случае переотправки в MBI, надо сначала раскостылить то, чтобы было накостылено в предыдущую отправку
        val isForceDc = feed.masterSystem == MasterSystem.MARKET && getForceDc(feed.url)
        feed.url = unFakeUrlIfNeeded(feed.url).trim()
        when (feed.source) {
            Source.FILE -> {
                val fileFeedInfo = FileFeedInfo(clientId, chiefUid, feed.id, feed.url, true)
                answer = marketClient.sendFileFeedToMbi(fileFeedInfo, enableDirectPlacement)
                fakeUrl = createFakeFeedUrl(answer.businessId, answer.shopId, answer.marketFeedId, feed.url)
                logger.info("File feed ${feed.id} was sent to MBI: shopId=${answer.shopId}, enabled=$enableDirectPlacement")
            }
            Source.SITE -> {
                val siteFeedInfo = toSiteFeedInfo(clientId, chiefUid, feed.id, feed.url)
                val (origSchema, origDomain) = getUrlParts(feed.url)
                answer = marketClient.sendSiteFeedToMbi(siteFeedInfo, enableDirectPlacement)
                fakeUrl = createFakeFeedUrl(
                    answer.businessId,
                    answer.shopId,
                    answer.marketFeedId,
                    origSchema.name.lowercase(),
                    origDomain
                )
                logger.info("Site feed ${feed.id} was sent to MBI: shop_id=${answer.shopId}, enabled=$enableDirectPlacement")
            }
            Source.URL -> {
                if (!isUrlInternal(feed.url)) {
                    val urlFeedInfo = UrlFeedInfo(clientId, chiefUid, feed.id, feed.url, feed.login, feed.plainPassword)
                    answer = marketClient.sendUrlFeedToMbi(urlFeedInfo, enableDirectPlacement)
                    fakeUrl = createFakeFeedUrl(
                        answer.businessId,
                        answer.shopId,
                        answer.marketFeedId,
                        feed.url,
                        feed.masterSystem == MasterSystem.MARKET,
                        isForceDc
                    )

                    // Если маркетный фид по магазину (по нескольким фидам из маркета с одним shopId) -
                    // проставляем marketFeedId = marketShopId (см. SyncMarketFeedsJob)
                    if (isForceDc) {
                        answer = MarketClient.SendFeedResult(answer.shopId, answer.shopId, answer.businessId)
                    }
                    logger.info("Url feed ${feed.id} was sent to MBI: shop_id=${answer.shopId}, enabled=$enableDirectPlacement")
                } else {
                    val fileFeedInfo = FileFeedInfo(clientId, chiefUid, feed.id, feed.url, true)
                    answer = marketClient.sendFileFeedToMbi(fileFeedInfo, enableDirectPlacement)
                    fakeUrl = createFakeFeedUrl(answer.businessId, answer.shopId, answer.marketFeedId, feed.url)
                    logger.info("Internal url feed ${feed.id} was sent to MBI: shop_id=${answer.shopId}, enabled=$enableDirectPlacement")
                }
            }
            else -> {
                throw IllegalStateException("Unexpected feed source value")
            }
        }
        val statusMbiEnabled = if (enableDirectPlacement) StatusMBIEnabled.YES else StatusMBIEnabled.NO

        val shard = shardSupport.getShardByClientIdStrictly(clientId)
        feedRepository.update(shard, listOf(ModelChanges(feed.id, Feed::class.java)
            .process(answer.marketFeedId, Feed.MARKET_FEED_ID)
            .process(answer.businessId, Feed.MARKET_BUSINESS_ID)
            .process(answer.shopId, Feed.MARKET_SHOP_ID)
            .process(fakeUrl, Feed.URL)
            .process(StatusMBISynced.YES, Feed.STATUS_MBI_SYNCED)
            .process(statusMbiEnabled, Feed.STATUS_MBI_ENABLED)
            .applyTo(feed as Feed)))

        return SendFeedResult(
            directFeedId = feed.id,
            marketFeedId = answer.marketFeedId,
            shopId = answer.shopId, 
            businessId = answer.businessId,
            fakeUrl = fakeUrl, 
            statusMBIEnabled = statusMbiEnabled
        )
    }

    private fun shouldBeEnabled(feed: FeedSimple) = !feed.usageTypes.isNullOrEmpty()
        || feed.lastUsed?.let { it.plus(FEED_KEEP_MBI_ENABLED_DURATION) >= LocalDateTime.now() } ?: true

    /**
     * Отправляет в MBI и информацию о том, как будет использован фид, в зависимости от типа группы, к которой
     * прикреплен фид, будут выбраны соответствующие пайплайны для офферов этого фида.
     * Если фид не был зарегистрирован в MBI, то сначала делает это
     */
    fun setFeedUsageType(clientId: ClientId, chiefUid: Long, feeds: List<Feed>, adGroupType: AdGroupType) {
        val toSendToMbi = feeds
            .filter { feed -> feed.masterSystem == MasterSystem.DIRECT && feed.marketShopId == null }
            .toList()
        for (feed in toSendToMbi) {
            sendFeed(clientId, chiefUid, feed)
        }
        feeds.forEach { feed ->
            val isGoodsAds = setOf(AdGroupType.DYNAMIC, AdGroupType.PERFORMANCE).contains(adGroupType)
            val isSearchSnippetGallery = adGroupType == BASE
            sendUsageTypesToMbi(feed.marketShopId, true, isGoodsAds, isSearchSnippetGallery)
        }
    }

    /**
     * Отправляет в MBI и информацию о том, как использован фид
     */
    fun setFeedUsageType(shopId: Long, usageTypes: Set<FeedUsageType>) {
        val isGoodsAds = usageTypes.contains(FeedUsageType.GOODS_ADS)
        val isSearchSnippetGallery = usageTypes.contains(FeedUsageType.SEARCH_SNIPPET_GALLERY)
        sendUsageTypesToMbi(shopId, true, isGoodsAds, isSearchSnippetGallery)
    }

    private fun sendUsageTypesToMbi(
        shopId: Long,
        isStandBy: Boolean,
        isGoodsAds: Boolean,
        isSearchSnippetGallery: Boolean
    ) {
        try {
            marketClient.setFeedFeaturesToMbi(
                FeedFeaturesInfo(
                    shopId,
                    isStandBy,
                    isGoodsAds,
                    isSearchSnippetGallery
                )
            )
            logger.info(
                "Features for feed with market_shop_id=$shopId were sent to MBI: standby=$isStandBy, " +
                    "searchSnippetGallery=$isSearchSnippetGallery, isGoodsAds=$isGoodsAds"
            )
        } catch (e: Exception) {
            logger.error(
                "Failed to send features for feed with market_shop_id=$shopId, features: standby=$isStandBy, " +
                    "searchSnippetGallery=$isSearchSnippetGallery, isGoodsAds=$isGoodsAds"
            )
        }
    }

    fun addSitePreview(ecomDomain: EcomDomain): AddSitePreviewResult {
        val sitePreviewInfo =
            SitePreviewInfo(toSchema(ecomDomain.schema), ecomDomain.domain, OFFERS_COUNT_FOR_PREVIEW)
        val answer = try {
            marketClient.addSitePreviewToMBI(sitePreviewInfo)
        } catch (e: Exception) {
            logger.error("Failed to add site preview for domain:${ecomDomain.domain}", e)
            throw e
        }
        logger.info(
            "Ecom domain ${ecomDomain.domain} with id=${ecomDomain.id} was registered in MBI " +
                "with shop_id ${answer.shopId} with offers count for preview: $OFFERS_COUNT_FOR_PREVIEW"
        )
        return AddSitePreviewResult(answer.marketFeedId, answer.shopId, answer.businessId)
    }

    fun deleteSitePreviews(ecomDomains: List<EcomDomain>): Int {
        val relaxedWorker = RelaxedWorker()
        return ecomDomains.asSequence()
            .filter { it.marketFeedId != null && it.marketShopId != null }
            .chunked(MAX_COUNT_OF_SITE_PREVIEWS_TO_DELETE)
            .map { ecomDomainsToDelete ->
                val deleteSitePreviewInfo = ecomDomainsToDelete.map { SitePreviewIds(it.marketShopId, it.marketFeedId) }
                val deletedSitePreviews = try {
                    relaxedWorker.callAndRelax<List<SitePreviewIds>, RuntimeException> {
                        marketClient.deleteSitePreviewsFromMBI(deleteSitePreviewInfo)
                    }.toSet()
                } catch (e: Exception) {
                    logger.error(
                        ecomDomainsInfoToString("Failed to delete site previews for ", ecomDomainsToDelete), e)
                    throw e
                }

                val (deletedEcomDomains, notDeletedEcomDomains) = ecomDomainsToDelete.partition {
                    deletedSitePreviews.contains(SitePreviewIds(it.marketShopId, it.marketFeedId))
                }

                if (deletedEcomDomains.isNotEmpty()) {
                    logger.info(
                        ecomDomainsInfoToString("Site previews was deleted from MBI for ", deletedEcomDomains))
                }
                if (notDeletedEcomDomains.isNotEmpty()) {
                    logger.warn(ecomDomainsInfoToString("Site previews wasn't deleted from MBI for ",
                        notDeletedEcomDomains))
                }

                deletedSitePreviews.size
            }
            .sum()
    }

    private fun toSiteFeedInfo(clientId: ClientId, chiefUid: Long, feedId: Long, url: String): SiteFeedInfo {
        val mainMirror: String?
        try {
            val mainMirrorByUrl = geminiClient.getMainMirrors(listOf(url))
            mainMirror = mainMirrorByUrl[url]
        } catch (e: GeminiClientException) {
            throw RuntimeException("Failed to get main mirror from gemini", e)
        }

        if (mainMirror == null) {
            throw RuntimeException("Failed to get main mirror from gemini")
        }

        val (schema, domain) = getUrlParts(mainMirror)
        return SiteFeedInfo(clientId, chiefUid, feedId, schema, domain)
    }

    private fun getUrlParts(url: String): UrlPartsResult {
        val parts = UrlParts.fromUrl(url)
        val schema = toSchema(parts.protocol)

        return UrlPartsResult(schema, parts.domain)
    }

    private fun toSchema(protocol: String) = if (protocol == "https") Schema.HTTPS else Schema.HTTP

    private fun ecomDomainsInfoToString(prefix: String, ecomDomains: List<EcomDomain>): String {
        return ecomDomains.joinToString(
            prefix = "$prefix[",
            postfix = "]"
        ) { "(id = ${it.id}, domain = ${it.domain}, shop_id = ${it.marketShopId}, feed_id = ${it.marketFeedId})" }
    }
}
