package ru.yandex.direct.core.entity.feed.validation;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.feed.container.FeedQueryFilter;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.feed.repository.FeedRepository;
import ru.yandex.direct.core.entity.uac.model.EcomDomain;
import ru.yandex.direct.core.entity.uac.service.EcomDomainsService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.utils.UrlUtils;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static java.util.Comparator.comparingLong;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.minBy;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedBySiteContainsDuplicatedUrl;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedBySiteForNotAllowedUrl;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedBySiteForNotDomainOnlyUrl;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedBySiteWithSameDomainAlreadyExists;
import static ru.yandex.direct.libs.mirrortools.utils.HostingsHandler.stripDomainTail;
import static ru.yandex.direct.libs.mirrortools.utils.HostingsHandler.stripProtocol;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class AbstractFeedValidationService {
    private static final Logger logger = LoggerFactory.getLogger(AbstractFeedValidationService.class);

    protected final FeedRepository feedRepository;
    protected final EcomDomainsService ecomDomainsService;

    protected AbstractFeedValidationService(FeedRepository feedRepository,
                                            EcomDomainsService ecomDomainsService) {
        this.feedRepository = feedRepository;
        this.ecomDomainsService = ecomDomainsService;
    }

    /**
     * Возвращает все дублирующиеся домены в рамках в рамках одного запроса добавления/обновления фида
     */
    protected static Set<String> getDuplicateDomains(List<String> feedsUrls,
                                                  Map<String, String> mainMirrorsBySiteFeedUrls) {
        Map<String, Long> domainRepeatsCount = StreamEx.of(feedsUrls)
                .map(feedUrl -> {
                    var mainMirror = mainMirrorsBySiteFeedUrls.get(feedUrl);
                    return getDomain(nvl(mainMirror, feedUrl));
                })
                .groupingBy(Function.identity(), Collectors.counting());

        return EntryStream.of(domainRepeatsCount)
                .filterValues(repeats -> repeats > 1)
                .keys()
                .toSet();
    }

    /**
     * Поиск существующих фидов по сайту с такими же доменами, что и у добавляемых/обновляемых фидовG
     * Проверка идет по полю target_domain
     */
    protected Map<String, Feed> findSiteFeedsWithSameDomain(int shard, ClientId clientId, List<String> feedsUrls,
                                                            Map<String, String> mainMirrorsBySiteFeedUrls) {
        if (feedsUrls.isEmpty()) {
            return Collections.emptyMap();
        }

        List<String> feedsUrlsToSearch = mapList(feedsUrls, feedUrl -> {
            var mainMirror = mainMirrorsBySiteFeedUrls.get(feedUrl);
            return getDomain(nvl(mainMirror, feedUrl));
        });

        FeedQueryFilter feedFilter = FeedQueryFilter.newBuilder()
                .withSources(List.of(Source.SITE))
                .withTargetDomains(feedsUrlsToSearch)
                .build();
        List<Feed> duplicateFeeds = feedRepository.get(shard, clientId, feedFilter);

        return StreamEx.of(duplicateFeeds)
                .mapToEntry(Feed::getTargetDomain, Function.identity())
                .mapKeys(AbstractFeedValidationService::getDomain)
                .grouping(collectingAndThen(minBy(comparingLong(Feed::getId)), o -> o.orElse(null)));
    }

    protected static Constraint<String, Defect> isDomainDuplicated(Map<String, Feed> existingSiteFeeds,
                                                                   Set<String> existingDomains,
                                                                   Map<String, String> mainMirrorsBySiteFeedUrls) {
        return feedUrl -> {
            String mainMirror = mainMirrorsBySiteFeedUrls.get(feedUrl);
            String feedDomain = getDomain(nvl(mainMirror, feedUrl));

            Feed duplicatedFeed = existingSiteFeeds.get(feedDomain);
            if (duplicatedFeed != null) {
                return feedBySiteWithSameDomainAlreadyExists(duplicatedFeed.getId());
            }

            if (existingDomains.contains(feedDomain)) {
                return feedBySiteContainsDuplicatedUrl(feedUrl);
            }

            return null;
        };
    }

    private static String getDomain(String url) {
        return stripDomainTail(stripProtocol(url)).trim();
    }

    protected static Constraint<String, Defect> isDomainAllowed(Set<String> notAllowedDomains,
                                                                Map<String, String> mainMirrorsBySiteFeedUrls) {
        return feedUrl -> {
            String feedMainMirror = nvl(mainMirrorsBySiteFeedUrls.get(feedUrl), feedUrl);
            String feedMainMirrorDomain = getDomain(feedMainMirror);
            if (notAllowedDomains.contains(feedMainMirrorDomain)) {
                return feedBySiteForNotAllowedUrl(feedUrl);
            }
            return null;
        };
    }

    protected static Constraint<String, Defect> isDomainOnlyUrl() {
        return feedUrl -> {
            var urlParts = UrlUtils.laxParseUrl(feedUrl);
            if (!isEmpty(urlParts.getPath()) && !urlParts.getPath().equals("/")) {
                return feedBySiteForNotDomainOnlyUrl(feedUrl, urlParts.toBuilder().withPath("").build().toUrl());
            }
            return null;
        };
    }

    protected static Constraint<String, Defect> isDomainSuitableEcom(Map<String, EcomDomain> ecomDomainsByFeedUrl,
                                                                     Boolean isCheckNeeded,
                                                                     Long minOffersCountAllowed,
                                                                     Long maxOffersCountAllowed) {
        return feedUrl -> {
            if (!isCheckNeeded) {
                return null;
            }
            var ecomDomain = ecomDomainsByFeedUrl.get(feedUrl);
            boolean notEcom = ecomDomain == null;
            if (notEcom || ecomDomain.getOffersCount() < minOffersCountAllowed
                    || ecomDomain.getOffersCount() > maxOffersCountAllowed) {
                logger.info("Site feed validation: can't add domain as ecom." +
                                "FeedUrl: {}, ecom: {}, offersCount: {}", feedUrl, !notEcom,
                        notEcom ? 0L : ecomDomain.getOffersCount());
                return feedBySiteForNotAllowedUrl(feedUrl);
            }
            return null;
        };
    }

    protected static boolean isFeedBySite(Feed feed) {
        return feed.getSource() == Source.SITE;
    }

}
