package ru.yandex.direct.core.entity.feed.validation;

import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.client.model.ClientLimits;
import ru.yandex.direct.core.entity.client.service.ClientLimitsService;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.model.MasterSystem;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.feed.repository.FeedRepository;
import ru.yandex.direct.core.entity.uac.model.EcomDomain;
import ru.yandex.direct.core.entity.uac.service.EcomDomainsService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.gemini.GeminiClient;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedInvalidFilename;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedInvalidHref;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedLoginCannotBeEmpty;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedLoginIsNotSet;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedNameCannotBeEmpty;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedPasswordCannotBeEmpty;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedPasswordIsNotSet;
import static ru.yandex.direct.core.entity.feed.validation.constraints.FeedConstraints.MBI_MAX_LOGIN_LENGTH;
import static ru.yandex.direct.core.entity.feed.validation.constraints.FeedConstraints.SOURCES_WITH_URL;
import static ru.yandex.direct.core.entity.feed.validation.constraints.FeedConstraints.notFakeUrl;
import static ru.yandex.direct.core.entity.feed.validation.constraints.FeedConstraints.validFeedFilename;
import static ru.yandex.direct.core.entity.feed.validation.constraints.FeedConstraints.validSource;
import static ru.yandex.direct.core.validation.defects.RightsDefects.noRights;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.unconditional;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.validEmail;
import static ru.yandex.direct.validation.constraint.StringConstraints.validUrl;
import static ru.yandex.direct.validation.defect.CollectionDefects.maxElementsExceeded;

@Service
public class AddFeedValidationService extends AbstractFeedValidationService {

    private final ClientLimitsService clientLimitsService;
    private final RbacService rbacService;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final GeminiClient geminiClient;
    private final EcomDomainsService ecomDomainsService;

    @Autowired
    public AddFeedValidationService(
            FeedRepository feedRepository,
            ClientLimitsService clientLimitsService,
            RbacService rbacService,
            PpcPropertiesSupport ppcPropertiesSupport,
            GeminiClient geminiClient,
            EcomDomainsService ecomDomainsService) {
        super(feedRepository, ecomDomainsService);
        this.clientLimitsService = clientLimitsService;
        this.rbacService = rbacService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.geminiClient = geminiClient;
        this.ecomDomainsService = ecomDomainsService;
    }

    public ValidationResult<List<Feed>, Defect> validate(
            int shard,
            ClientId clientId,
            Long clientUid,
            Long operatorUid,
            List<Feed> feeds,
            boolean isSiteFeedsAllowed) {
        ListValidationBuilder<Feed, Defect> lvb = ListValidationBuilder.of(feeds);
        boolean canWrite = rbacService.canWrite(operatorUid, clientUid);
        lvb.check(unconditional(noRights()), When.isFalse(canWrite));
        if (lvb.getResult().hasAnyErrors()) {
            return lvb.getResult();
        }
        int blFeedsCount = feedRepository.getBlCount(shard, clientId);
        ClientLimits clientLimits = clientLimitsService.getClientLimits(clientId);
        Long feedCountLimit = clientLimits.getFeedCountLimitOrDefault();

        Map<String, Feed> existingSiteFeeds = Collections.emptyMap();
        Set<String> existingDomains = Collections.emptySet();
        List<String> siteFeedsUrls = getSiteFeedsUrls(feeds);
        var mainMirrorsBySiteFeedUrls = geminiClient.getMainMirrors(siteFeedsUrls);
        if (isSiteFeedsAllowed) {
            existingSiteFeeds = findSiteFeedsWithSameDomain(shard, clientId, siteFeedsUrls, mainMirrorsBySiteFeedUrls);
            existingDomains = getDuplicateDomains(siteFeedsUrls, mainMirrorsBySiteFeedUrls);
        }
        Set<String> notAllowedDomains = nvl(ppcPropertiesSupport
                .get(PpcPropertyNames.DOMAINS_NOT_ALLOWED_FOR_FEED_FROM_SITE, Duration.ofMinutes(5))
                .get(), emptySet());
        var ecomDomainsBySiteFeedUrls = ecomDomainsService.getEcomDomainsByUrls(siteFeedsUrls);
        return lvb
                .check(maxFeedsLimit(blFeedsCount, feedCountLimit))
                .checkEachBy(validateFeed(isSiteFeedsAllowed, existingSiteFeeds, existingDomains, notAllowedDomains,
                                mainMirrorsBySiteFeedUrls, ecomDomainsBySiteFeedUrls),
                        When.isValid())
                .getResult();
    }

    /**
     * В текущем виде этот метод нужен только на переходный период, пока часть фидов продолжают грузиться через BL.
     * Когда весь парсинг и валидация фидов уедут в DataCamp, валидацию на количество фидов нужно будет или переделать
     * или совсем убрать.
     */
    private Constraint<List<Feed>, Defect> maxFeedsLimit(int blFeedsCount,
                                                         Long maxFeedsLimit) {
        return feedsToAdd -> {
            long newBlFeedsCount = blFeedsCount
                    + feedsToAdd.stream()
                    .filter(this::isLoadingByBl)
                    .count();
            if (newBlFeedsCount <= maxFeedsLimit) {
                return null;
            }
            return maxElementsExceeded(maxFeedsLimit.intValue());
        };
    }

    private boolean isLoadingByBl(Feed feed) {
        return feed.getMasterSystem() == null || feed.getMasterSystem() == MasterSystem.DIRECT;
    }

    private Validator<Feed, Defect> validateFeed(boolean isSiteFeedsAllowed,
                                                 Map<String, Feed> existingSiteFeeds,
                                                 Set<String> existingDomains,
                                                 Set<String> notAllowedDomains,
                                                 Map<String, String> mainMirrorsBySiteFeedUrls,
                                                 Map<String, EcomDomain> ecomDomainsBySiteFeedUrls) {
        return feed -> {
            ModelItemValidationBuilder<Feed> vb = ModelItemValidationBuilder.of(feed);
            vb.item(Feed.BUSINESS_TYPE)
                    .check(notNull());
            vb.item(Feed.SOURCE)
                    .check(notNull())
                    .check(validSource(isSiteFeedsAllowed, feed.getBusinessType()), When.notNull());
            vb.item(Feed.NAME)
                    .check(notNull())
                    .check(notBlank(), feedNameCannotBeEmpty());
            vb.item(Feed.URL)
                    .check(notNull(), When.isTrue(SOURCES_WITH_URL.contains(feed.getSource())))
                    .check(validUrl(), feedInvalidHref())
                    .check(notFakeUrl(), When.isValidAnd(When.isTrue(feed.getMasterSystem() == MasterSystem.DIRECT)))
                    .check(isDomainDuplicated(existingSiteFeeds, existingDomains, mainMirrorsBySiteFeedUrls),
                            When.isValidAnd(When.isTrue(isSiteFeedsAllowed && isFeedBySite(feed))))
                    .check(isDomainAllowed(notAllowedDomains, mainMirrorsBySiteFeedUrls),
                            When.isValidAnd(When.isTrue(isSiteFeedsAllowed && isFeedBySite(feed))))
                    .check(isDomainOnlyUrl(), When.isTrue(isSiteFeedsAllowed && isFeedBySite(feed)))
                    .check(isDomainSuitableEcom(ecomDomainsBySiteFeedUrls,
                                    ecomDomainsService.isDomainSuitableEcomCheckNeeded(),
                                    ecomDomainsService.getMinOffersForEcomAllowed(),
                                    ecomDomainsService.getMaxOffersForEcomAllowed()),
                            When.isTrue(isSiteFeedsAllowed && isFeedBySite(feed)));

            vb.item(Feed.FILE_DATA)
                    .check(notNull(), When.isTrue(feed.getSource() == Source.FILE));
            vb.item(Feed.FILENAME)
                    .check(validFeedFilename(), feedInvalidFilename(), When.notNull());
            vb.item(Feed.LOGIN)
                    .check(notNull(), feedLoginIsNotSet(), When.isTrue(feed.getPlainPassword() != null))
                    .check(notBlank(), feedLoginCannotBeEmpty(), When.isValidAnd(When.notNull()))
                    .check(maxStringLength(MBI_MAX_LOGIN_LENGTH));
            vb.item(Feed.PLAIN_PASSWORD)
                    .check(notNull(), feedPasswordIsNotSet(), When.isTrue(feed.getLogin() != null))
                    .check(notBlank(), feedPasswordCannotBeEmpty(), When.isValidAnd(When.notNull()));
            vb.item(Feed.EMAIL)
                    .check(validEmail(), When.notNull());
            vb.item(Feed.SHOP_NAME)
                    .check(notNull(), When.isTrue(feed.getMasterSystem() == MasterSystem.MARKET))
                    .check(notBlank(), When.isValidAnd(When.isTrue(feed.getMasterSystem() == MasterSystem.MARKET)));
            return vb.getResult();
        };
    }

    private List<String> getSiteFeedsUrls(List<Feed> feeds) {
        return filterAndMapList(feeds, feed -> feed.getSource() == Source.SITE, Feed::getUrl);
    }

}
