package ru.yandex.direct.core.entity.feed.validation;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.feed.model.UpdateStatus;
import ru.yandex.direct.core.entity.feed.repository.FeedRepository;
import ru.yandex.direct.core.entity.uac.service.GrutUacCampaignService;
import ru.yandex.direct.dbschema.ppc.enums.FeedsUpdateStatus;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.feedNotExist;
import static ru.yandex.direct.core.validation.defects.RightsDefects.noRights;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.unconditional;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedObject;

@Service
public class DeleteFeedValidationService {

    private final FeedRepository feedRepository;
    private final AdGroupRepository adGroupRepository;
    private final RbacService rbacService;
    private final GrutUacCampaignService grutUacCampaignService;

    @Autowired
    public DeleteFeedValidationService(
            FeedRepository feedRepository,
            AdGroupRepository adGroupRepository,
            RbacService rbacService,
            @Lazy GrutUacCampaignService grutUacCampaignService
    ) {
        this.feedRepository = feedRepository;
        this.adGroupRepository = adGroupRepository;
        this.rbacService = rbacService;
        this.grutUacCampaignService = grutUacCampaignService;
    }

    public ValidationResult<List<Long>, Defect> validate(
            int shard,
            ClientId clientId,
            Long clientUid,
            Long operatorUid,
            List<Long> feedIds
    ) {
        ListValidationBuilder<Long, Defect> lvb = ListValidationBuilder.of(feedIds);
        boolean canWrite = rbacService.canWrite(operatorUid, clientUid);
        lvb.check(unconditional(noRights()), When.isFalse(canWrite));
        if (lvb.getResult().hasAnyErrors()) {
            return lvb.getResult();
        }
        Map<Long, FeedsUpdateStatus> statusesById = feedRepository.getStatusesById(shard, clientId, feedIds);
        List<Long> clientFeedIds = new ArrayList<>(statusesById.keySet());
        Map<Long, List<Long>> adGroupIdsByFeedId = adGroupRepository.getAdGroupIdsByFeedId(shard, clientFeedIds);
        Set<Long> briefsFeedIds = grutUacCampaignService.getCampaignsByFeedIds(clientId.toString(), feedIds).keySet();
        return lvb.checkEach(notNull())
                .checkEach(validId(), When.isValid())
                .checkEach(unique(), duplicatedObject(), When.isValid())
                .checkEachBy(usageFeedValidator(statusesById, adGroupIdsByFeedId.keySet(), briefsFeedIds),
                        When.isValid())
                .getResult();
    }

    private Validator<Long, Defect> usageFeedValidator(
            Map<Long, FeedsUpdateStatus> feedStatusesById,
            Set<Long> usedFeedIds,
            Set<Long> briefsFeedIds
    ) {
        return feedId -> {
            FeedsUpdateStatus feedStatus = feedStatusesById.get(feedId);
            if (feedStatus == null) {
                return ValidationResult.failed(feedId, feedNotExist(feedId));
            }
            UpdateStatus updateStatus = UpdateStatus.fromSource(feedStatus);
            if (updateStatus != UpdateStatus.DONE && updateStatus != UpdateStatus.ERROR) {
                //noinspection ConstantConditions
                return ValidationResult.failed(feedId, FeedDefects.feedStatusWrong(updateStatus));
            }
            if (usedFeedIds.contains(feedId)) {
                return ValidationResult.failed(feedId, FeedDefects.feedUsedInAdGroup());
            }
            if (briefsFeedIds.contains(feedId)) {
                return ValidationResult.failed(feedId, FeedDefects.feedIsUsedInCampaignBrief());
            }
            return ValidationResult.success(feedId);
        };
    }
}
