package ru.yandex.direct.core.entity.feed.validation;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.feed.container.FeedQueryFilter;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.model.MasterSystem;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.feed.repository.FeedRepository;
import ru.yandex.direct.core.entity.uac.model.EcomDomain;
import ru.yandex.direct.core.entity.uac.service.EcomDomainsService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.gemini.GeminiClient;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedApartResetLoginAnPassword;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedInconsistentTypeForUpdate;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedInvalidHref;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedLoginCannotBeEmpty;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedNameCannotBeEmpty;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedNotExist;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedPasswordCannotBeEmpty;
import static ru.yandex.direct.core.entity.feed.validation.constraints.FeedConstraints.MBI_MAX_LOGIN_LENGTH;
import static ru.yandex.direct.core.entity.feed.validation.constraints.FeedConstraints.SOURCES_WITH_URL;
import static ru.yandex.direct.core.entity.feed.validation.constraints.FeedConstraints.notFakeUrl;
import static ru.yandex.direct.core.entity.feed.validation.constraints.FeedConstraints.validFeedFilename;
import static ru.yandex.direct.core.validation.defects.RightsDefects.noRights;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.unconditional;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.validUrl;

@Service
public class UpdateFeedValidationService extends AbstractFeedValidationService {
    private final ModelChangesValidationTool updateValidationTool;
    private final RbacService rbacService;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final GeminiClient geminiClient;
    private final EcomDomainsService ecomDomainsService;

    @Autowired
    public UpdateFeedValidationService(FeedRepository feedRepository, RbacService rbacService,
                                       PpcPropertiesSupport ppcPropertiesSupport, GeminiClient geminiClient,
                                       EcomDomainsService ecomDomainsService) {
        super(feedRepository, ecomDomainsService);
        this.rbacService = rbacService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.geminiClient = geminiClient;
        this.ecomDomainsService = ecomDomainsService;
        updateValidationTool = ModelChangesValidationTool.builder()
                .objectNotFoundDefect(feedNotExist())
                .build();
    }

    public ValidationResult<List<ModelChanges<Feed>>, Defect> validate(
            int shard,
            ClientId clientId,
            Long clientUid,
            Long operatorUid,
            List<ModelChanges<Feed>> modelChanges
    ) {
        ListValidationBuilder<ModelChanges<Feed>, Defect> lvb = ListValidationBuilder.of(modelChanges, Defect.class);
        boolean canWrite = rbacService.canWrite(operatorUid, clientUid);
        lvb.check(unconditional(noRights()), When.isFalse(canWrite));
        if (lvb.getResult().hasAnyErrors()) {
            return lvb.getResult();
        }
        List<Long> feedIds = mapList(modelChanges, ModelChanges::getId);
        FeedQueryFilter feedQueryFilter = FeedQueryFilter.newBuilder().withFeedIds(feedIds).build();
        List<Feed> existingFeeds = feedRepository.get(shard, clientId, feedQueryFilter);
        Map<Long, Feed> existingFeedsById = StreamEx.of(existingFeeds).toMap(Feed::getId, Function.identity());
        ValidationResult<List<ModelChanges<Feed>>, Defect> vr =
                updateValidationTool.validateModelChangesList(modelChanges, existingFeedsById.keySet());
        if (vr.hasAnyErrors()) {
            return vr;
        }

        List<String> siteFeedsUrls = getSiteFeedsUrls(modelChanges, existingFeedsById);
        var mainMirrorsBySiteFeedUrls = geminiClient.getMainMirrors(siteFeedsUrls);
        Map<String, Feed> existingSiteFeeds = filterUpdatingFeedsFromDuplicates(
                findSiteFeedsWithSameDomain(shard, clientId, siteFeedsUrls, mainMirrorsBySiteFeedUrls), existingFeedsById.keySet());
        Set<String> existingDomains = getDuplicateDomains(siteFeedsUrls, mainMirrorsBySiteFeedUrls);
        Set<String> notAllowedDomains = nvl(ppcPropertiesSupport
                .get(PpcPropertyNames.DOMAINS_NOT_ALLOWED_FOR_FEED_FROM_SITE, Duration.ofMinutes(5))
                .get(), emptySet());
        var ecomDomainsBySiteFeedUrls = ecomDomainsService.getEcomDomainsByUrls(siteFeedsUrls);

        lvb.checkEachBy(mc -> validateChanges(mc, existingFeedsById.get(mc.getId()), existingSiteFeeds,
                        existingDomains, notAllowedDomains, mainMirrorsBySiteFeedUrls, ecomDomainsBySiteFeedUrls),
                When.isValid());
        return lvb.getResult();
    }

    private ValidationResult<ModelChanges<Feed>, Defect> validateChanges(ModelChanges<Feed> modelChanges, Feed feed,
                                                                         Map<String, Feed> existingSiteFeeds,
                                                                         Set<String> existingDomains,
                                                                         Set<String> notAllowedDomains,
                                                                         Map<String, String> mainMirrorsBySiteFeedUrls,
                                                                         Map<String, EcomDomain> ecomDomainsBySiteFeedUrls) {
        ItemValidationBuilder<ModelChanges<Feed>, Defect> vb = ItemValidationBuilder.of(modelChanges);
        if (modelChanges.isPropChanged(Feed.NAME)) {
            vb.item(modelChanges.getChangedProp(Feed.NAME), Feed.NAME.name())
                    .check(notNull())
                    .check(notBlank(), feedNameCannotBeEmpty(), When.isValid());
        }
        Source source = feed.getSource();
        if (modelChanges.isPropChanged(Feed.URL)) {
            //noinspection unchecked
            vb.item(modelChanges.getChangedProp(Feed.URL), Feed.URL.name())
                    .check(unconditional(feedInconsistentTypeForUpdate()),
                            When.isFalse(SOURCES_WITH_URL.contains(source)))
                    .check(notNull(), When.isValid())
                    .check(validUrl(), feedInvalidHref(), When.isValid())
                    .check(notFakeUrl(), When.isValidAnd(When.isTrue(feed.getMasterSystem() == MasterSystem.DIRECT)))
                    .check(isDomainAllowed(notAllowedDomains, mainMirrorsBySiteFeedUrls),
                            When.isValidAnd(When.isTrue(isFeedBySite(feed))))
                    .check(isDomainDuplicated(existingSiteFeeds, existingDomains, mainMirrorsBySiteFeedUrls),
                            When.isValidAnd(When.isTrue(isFeedBySite(feed))))
                    .check(isDomainOnlyUrl(), When.isTrue(isFeedBySite(feed)))
                    .check(isDomainSuitableEcom(ecomDomainsBySiteFeedUrls,
                                    ecomDomainsService.isDomainSuitableEcomCheckNeeded(),
                                    ecomDomainsService.getMinOffersForEcomAllowed(),
                                    ecomDomainsService.getMaxOffersForEcomAllowed()),
                            When.isTrue(isFeedBySite(feed)));
        }
        if (modelChanges.isPropChanged(Feed.FILENAME)) {
            //noinspection unchecked
            vb.item(modelChanges.getChangedProp(Feed.FILENAME), Feed.FILENAME.name())
                    .check(unconditional(feedInconsistentTypeForUpdate()), When.isFalse(source == Source.FILE))
                    .check(validFeedFilename(), When.isValidAnd(When.notNull()));
        }
        if (modelChanges.isPropChanged(Feed.LOGIN)) {
            validateNewLogin(modelChanges, vb, source);
        }
        if (modelChanges.isPropChanged(Feed.PLAIN_PASSWORD)) {
            validateNewPassword(modelChanges, vb, source);
        }
        if (modelChanges.isPropChanged(Feed.SHOP_NAME)) {
            MasterSystem masterSystem = feed.getMasterSystem();
            vb.item(modelChanges.getChangedProp(Feed.SHOP_NAME), Feed.SHOP_NAME.name())
                    .check(notNull(), When.isTrue(masterSystem == MasterSystem.MARKET))
                    .check(notBlank(), When.isValidAnd(When.isTrue(masterSystem == MasterSystem.MARKET)));
        }
        return vb.getResult();
    }

    private void validateNewLogin(
            ModelChanges<Feed> modelChanges,
            ItemValidationBuilder<ModelChanges<Feed>, Defect> vb,
            Source source
    ) {
        ItemValidationBuilder<String, Defect> item =
                vb.item(modelChanges.getChangedProp(Feed.LOGIN), Feed.LOGIN.name());
        //noinspection unchecked
        item.check(unconditional(feedInconsistentTypeForUpdate()), When.isFalse(source == Source.URL));
        if (item.getResult().hasAnyErrors()) {
            return;
        }
        item.check(notBlank(), feedLoginCannotBeEmpty(), When.isValidAnd(When.notNull()));
        item.check(maxStringLength(MBI_MAX_LOGIN_LENGTH));
    }

    private void validateNewPassword(
            ModelChanges<Feed> modelChanges,
            ItemValidationBuilder<ModelChanges<Feed>, Defect> vb,
            Source source
    ) {
        ItemValidationBuilder<String, Defect> item =
                vb.item(modelChanges.getChangedProp(Feed.PLAIN_PASSWORD), Feed.PLAIN_PASSWORD.name());
        //noinspection unchecked
        item.check(unconditional(feedInconsistentTypeForUpdate()), When.isFalse(source == Source.URL));
        if (item.getResult().hasAnyErrors()) {
            return;
        }
        item.check(notBlank(), feedPasswordCannotBeEmpty(), When.isValidAnd(When.notNull()));
    }

    public ValidationResult<List<Feed>, Defect> validateAppliedChanges(
            ValidationResult<List<Feed>, Defect> validationResult
    ) {
        return new ListValidationBuilder<>(validationResult)
                .checkBy(this::validateAuth, When.isValid())
                .getResult();
    }

    private ValidationResult<List<Feed>, Defect> validateAuth(List<Feed> feeds) {
        ListValidationBuilder<Feed, Defect> lvb = ListValidationBuilder.of(feeds);
        return lvb.checkEachBy(validateFeed())
                .getResult();
    }

    private Validator<Feed, Defect> validateFeed() {
        return feed -> {
            ModelItemValidationBuilder<Feed> vb = ModelItemValidationBuilder.of(feed);
            vb.item(Feed.LOGIN)
                    .check(notNull(), feedApartResetLoginAnPassword(), When.isTrue(feed.getPlainPassword() != null));
            vb.item(Feed.PLAIN_PASSWORD)
                    .check(notNull(), feedApartResetLoginAnPassword(), When.isTrue(feed.getLogin() != null));
            return vb.getResult();
        };
    }

    private List<String> getSiteFeedsUrls(List<ModelChanges<Feed>> modelChanges, Map<Long, Feed> existingFeedsById) {
        return StreamEx.of(modelChanges)
                .filter(mc -> mc.isPropChanged(Feed.URL))
                .filter(mc -> existingFeedsById.get(mc.getId()).getSource() == Source.SITE)
                .map(mc -> mc.getChangedProp(Feed.URL))
                .toList();
    }

    /**
     * Убрать из дубликатов те фиды, которые обновляются в рамках одного запроса
     */
    private Map<String, Feed> filterUpdatingFeedsFromDuplicates(Map<String, Feed> existingSiteFeeds,
                                                                Set<Long> updatingFeedsIds) {
        return EntryStream.of(existingSiteFeeds)
                .filterValues(feed -> !updatingFeedsIds.contains(feed.getId()))
                .toMap();
    }

}
