package ru.yandex.direct.core.entity.feed.validation.constraints;

import java.util.List;
import java.util.Set;
import java.util.function.Predicate;

import org.apache.commons.io.FilenameUtils;

import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static ru.yandex.direct.core.entity.feed.FeedUtilsKt.isFakeUrl;
import static ru.yandex.direct.core.entity.feed.model.Source.SITE;
import static ru.yandex.direct.core.entity.feed.model.Source.URL;
import static ru.yandex.direct.core.entity.feed.validation.FeedDefects.feedInvalidFilename;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

public class FeedConstraints {
    private static final List<String> FILE_EXTENSIONS = List.of("xml", "csv", "tsv", "yml", "xls", "xlsx", "gz", "zip");
    public static final Set<Source> SOURCES_WITH_URL = Set.of(URL, SITE);
    public static final int MBI_MAX_LOGIN_LENGTH = 50; // Ограничение зависит от MBI. См. MBI-82164

    public static Constraint<String, Defect> validFeedFilename() {
        return fromPredicate(FeedConstraints::isValidFilename, feedInvalidFilename());
    }

    private static boolean isValidFilename(String filename) {
        String extension = FilenameUtils.getExtension(filename).toLowerCase();
        return FILE_EXTENSIONS.contains(extension);
    }

    public static Constraint<String, Defect> notFakeUrl() {
        Predicate<String> predicate =
                url -> url == null || !isFakeUrl(url);
        return fromPredicate(predicate, invalidValue());
    }

    public static Constraint<Source, Defect> validSource(boolean isSiteFeedsAllowed,
                                                         BusinessType feedBusinessType) {
        return fromPredicate(
                source -> isSourceValid(isSiteFeedsAllowed, feedBusinessType, source),
                invalidValue());
    }

    private static boolean isSourceValid(boolean isSiteFeedsAllowed, BusinessType feedBusinessType, Source source) {
        if (source != SITE) {
            return true;
        }
        return isSiteFeedsAllowed && BusinessType.RETAIL == feedBusinessType;
    }

}
