package ru.yandex.direct.core.entity.feedfilter.converter

import com.fasterxml.jackson.annotation.JsonProperty
import com.fasterxml.jackson.annotation.JsonPropertyOrder
import com.fasterxml.jackson.core.JsonGenerator
import com.fasterxml.jackson.core.JsonToken
import com.fasterxml.jackson.databind.ObjectMapper
import com.fasterxml.jackson.databind.SerializerProvider
import com.fasterxml.jackson.databind.annotation.JsonDeserialize
import com.fasterxml.jackson.databind.annotation.JsonSerialize
import com.fasterxml.jackson.databind.jsontype.TypeSerializer
import com.fasterxml.jackson.databind.module.SimpleModule
import com.fasterxml.jackson.databind.ser.std.StdSerializer
import com.fasterxml.jackson.databind.util.StdConverter
import ru.yandex.direct.core.entity.feedfilter.model.FeedFilter
import ru.yandex.direct.core.entity.feedfilter.model.FeedFilterCondition
import ru.yandex.direct.core.entity.feedfilter.model.FeedFilterTab
import ru.yandex.direct.core.entity.feedfilter.service.FeedFilterByMapperConverter
import ru.yandex.direct.core.entity.feedfilter.service.FeedFilterConverter
import ru.yandex.direct.core.entity.feedfilter.service.FeedFilterFieldValueByMapperSerializer
import ru.yandex.direct.core.entity.feedfilter.service.FeedFilterFieldValueSerializer
import ru.yandex.direct.core.entity.feedfilter.service.ParsedFeedFilterConditionAutoSerializingFactory
import ru.yandex.direct.core.entity.feedfilter.service.ParsedFeedFilterConditionFactory
import ru.yandex.direct.core.entity.performancefilter.container.DecimalRange
import ru.yandex.direct.core.entity.performancefilter.container.Exists
import ru.yandex.direct.core.entity.performancefilter.model.Operator
import ru.yandex.direct.utils.JsonUtils
import java.math.RoundingMode
import java.text.DecimalFormat

class FeedFilterConverters private constructor() {
    companion object {
        @JvmField
        val DB_VALUE_SERIALIZER: FeedFilterFieldValueSerializer =
            FeedFilterFieldValueByMapperSerializer(FeedFilterGridAndDbCommonFormat.MAPPER)

        @JvmField
        val DB_CONVERTER: FeedFilterConverter =
            FeedFilterByMapperConverter(FeedFilterGridAndDbCommonFormat.MAPPER)

        @JvmField
        val DB_CONDITION_FACTORY: ParsedFeedFilterConditionFactory =
            ParsedFeedFilterConditionAutoSerializingFactory(DB_VALUE_SERIALIZER)
    }
}

internal object FeedFilterGridAndDbCommonFormat {

    val MAPPER: ObjectMapper = JsonUtils.MAPPER.copy().also { applyToMapper(it) }

    private fun applyToMapper(mapper: ObjectMapper) {
        @JsonSerialize(converter = FeedFilterSerializationConverter::class)
        @JsonDeserialize(converter = FeedFilterDeserializationConverter::class)
        class FeedFilterMixin

        @JsonSerialize(converter = DecimalRangeSerializationConverter::class)
        class DecimalRangeMixin

        @JsonDeserialize(converter = ExistsDeserializationConverter::class)
        class ExistsMixin

        mapper.apply {
            addMixIn(FeedFilter::class.java, FeedFilterMixin::class.java)
            addMixIn(DecimalRange::class.java, DecimalRangeMixin::class.java)
            addMixIn(Exists::class.java, ExistsMixin::class.java)

            registerModule(SimpleModule().apply {
                addSerializer(ExistsSerializer)
                addSerializer(NumberSerializer)
            })
        }
    }

    private object DecimalRangeSerializationConverter : StdConverter<DecimalRange, String>() {
        override fun convert(value: DecimalRange): String =
            value.left?.toString().orEmpty() + DecimalRange.SEPARATOR + value.right?.toString().orEmpty()
    }

    private object ExistsDeserializationConverter : StdConverter<Int, Exists>() {
        override fun convert(value: Int): Exists = Exists(value == 1)
    }

    private object ExistsSerializer : StdSerializer<Exists>(Exists::class.java) {
        override fun serialize(value: Exists, gen: JsonGenerator, provider: SerializerProvider) {
            gen.writeNumber(if (value.value()) 1 else 0)
        }

        override fun serializeWithType(
            value: Exists,
            gen: JsonGenerator,
            serializers: SerializerProvider,
            typeSer: TypeSerializer
        ) {
            val typeId = typeSer.typeId(value, JsonToken.VALUE_NUMBER_INT)
            typeSer.writeTypePrefix(gen, typeId)
            serialize(value, gen, serializers)
            typeSer.writeTypeSuffix(gen, typeId)
        }
    }

    private object NumberSerializer : StdSerializer<Double>(Double::class.javaObjectType) {
        private val decimalFormat = DecimalFormat("#.##").apply { roundingMode = RoundingMode.DOWN }

        override fun serialize(value: Double, gen: JsonGenerator, serializers: SerializerProvider?) {
            gen.writeNumber(decimalFormat.format(value))
        }
    }


    @JsonPropertyOrder("fromTab", "conditions")
    private data class FeedFilterDbData(
        @JsonProperty("fromTab")
        val fromTab: FeedFilterTab,

        @JsonProperty("conditions")
        @JsonSerialize(contentConverter = FeedFilterConditionSerializationConverter::class)
        @JsonDeserialize(contentConverter = FeedFilterConditionDeserializationConverter::class)
        val conditions: List<FeedFilterCondition<*>>
    )

    private class FeedFilterSerializationConverter : StdConverter<FeedFilter, FeedFilterDbData>() {
        override fun convert(value: FeedFilter) = FeedFilterDbData(
            fromTab = value.tab,
            conditions = value.conditions
        )
    }

    private class FeedFilterDeserializationConverter : StdConverter<FeedFilterDbData, FeedFilter>() {
        override fun convert(value: FeedFilterDbData): FeedFilter = FeedFilter()
            .withTab(value.fromTab)
            .withConditions(value.conditions)
    }

    @JsonPropertyOrder("fieldName", "operator", "value")
    private data class FeedFilterConditionDbData(
        @JsonProperty("fieldName")
        val fieldName: String,

        @JsonProperty("operator")
        val operator: Operator,

        @JsonProperty("value")
        val value: FeedFilterConditionValue
    )

    private class FeedFilterConditionSerializationConverter :
        StdConverter<FeedFilterCondition<*>, FeedFilterConditionDbData>() {
        override fun convert(condition: FeedFilterCondition<*>) = FeedFilterConditionDbData(
            fieldName = condition.fieldName,
            operator = condition.operator,
            value = FeedFilterConditionValue.fromObject(condition.parsedValue!!)
        )
    }

    private class FeedFilterConditionDeserializationConverter :
        StdConverter<FeedFilterConditionDbData, FeedFilterCondition<*>>() {
        override fun convert(conditionData: FeedFilterConditionDbData) =
            FeedFilterConverters.DB_CONDITION_FACTORY.create(
                conditionData.fieldName,
                conditionData.operator,
                conditionData.value.toObject()
            )
    }
}
