package ru.yandex.direct.core.entity.feedoffer.converter

import Market.DataCamp.DataCampContractTypes.Currency
import ru.yandex.direct.core.entity.feedoffer.model.FeedOffer
import ru.yandex.direct.currency.CurrencyCode
import java.io.StringWriter
import java.time.LocalDateTime
import java.time.ZoneId
import java.time.format.DateTimeFormatter
import java.time.temporal.ChronoUnit
import javax.xml.bind.JAXBContext
import javax.xml.bind.Marshaller
import javax.xml.bind.annotation.XmlAccessType
import javax.xml.bind.annotation.XmlAccessorType
import javax.xml.bind.annotation.XmlAttribute
import javax.xml.bind.annotation.XmlElement
import javax.xml.bind.annotation.XmlRootElement
import javax.xml.bind.annotation.XmlValue

/**
 * Описывает структуру офферов в формате YML:
 * https://yandex.ru/support/partnermarket/export/yml.html#yml-format
 */
@XmlRootElement(name = "yml_catalog")
@XmlAccessorType(XmlAccessType.FIELD)
data class YmlCatalog(
    @field:XmlAttribute(name = "date") var date: String? = null,
    @field:XmlElement(name = "shop") var shop: FeedShop = FeedShop()
)

@XmlAccessorType(XmlAccessType.FIELD)
data class FeedShop(
    @field:XmlElement(name = "name") var name: String? = null,
    @field:XmlElement(name = "company") var company: String? = null,
    @field:XmlElement(name = "url") var url: String? = null,
    @field:XmlElement(name = "currencies") var currencies: FeedCurrencies = FeedCurrencies(),
    @field:XmlElement(name = "categories") var categories: FeedCategories = FeedCategories(),
    @field:XmlElement(name = "offers") var offers: FeedOffersData = FeedOffersData()
)

@XmlAccessorType(XmlAccessType.FIELD)
data class FeedCurrencies(
    @field:XmlElement(name = "currency") var currencies: List<FeedCurrency> = emptyList()
)

@XmlAccessorType(XmlAccessType.FIELD)
class FeedCurrency(
    @field:XmlAttribute(name = "id") var id: String = CurrencyCode.RUB.name,
    @field:XmlAttribute(name = "rate") var rate: String = "1"
)

@XmlAccessorType(XmlAccessType.FIELD)
data class FeedCategories(
    @field:XmlElement(name = "category") var categories: List<FeedCategory> = emptyList()
)

@XmlAccessorType(XmlAccessType.FIELD)
data class FeedCategory(
    @field:XmlAttribute(name = "id") var id: Long = 1,
    @field:XmlAttribute(name = "parentId") var parentId: Long? = null,
    @field:XmlValue var name: String = ""
)

@XmlAccessorType(XmlAccessType.FIELD)
data class FeedOffersData(
    @field:XmlElement(name = "offer") var offers: List<FeedOfferData> = emptyList()
)

@XmlAccessorType(XmlAccessType.FIELD)
data class FeedOfferData(
    @field:XmlAttribute(name = "id") var id: Long = 0,
    @field:XmlAttribute(name = "available") var available: Boolean = false,
    @field:XmlElement(name = "url") var url: String? = null,
    @field:XmlElement(name = "price") var price: String? = null,
    @field:XmlElement(name = "oldprice") var oldPrice: String? = null,
    @field:XmlElement(name = "currencyId") var currencyId: String? = null,
    @field:XmlElement(name = "categoryId") var categoryId: Long? = null,
    @field:XmlElement(name = "picture") var pictures: List<String> = emptyList(),
    @field:XmlElement(name = "name") var name: String? = null,
    @field:XmlElement(name = "description") var description: String? = null,
    @field:XmlElement(name = "vendor") var vendor: String? = null,
    @field:XmlElement(name = "model") var model: String? = null
)

object FeedOffersToYmlConverter {
    private const val DEFAULT_CATEGORY = ""

    private fun currentDateTime() = LocalDateTime.now()
        .truncatedTo(ChronoUnit.SECONDS)
        .atZone(ZoneId.systemDefault())
        .format(DateTimeFormatter.ISO_OFFSET_DATE_TIME)

    /**
     * Возвращает основную валюту, в которой будут отображаться цены в офферах
     * https://yandex.ru/support/partnermarket/elements/currencies.html
     */
    private fun getFeedCurrencies(currencies: Collection<Currency>): FeedCurrencies {
        if (currencies.isEmpty()) {
            return FeedCurrencies(emptyList())
        }
        // Выбираем основной валютой фида - наиболее часто встречающуюся
        val currencyCode = currencies.groupingBy { it }
            .eachCount()
            .maxByOrNull { it.value }!!
            .key
        return FeedCurrencies(listOf(FeedCurrency(currencyCode.name, "1")))
    }

    /**
     * Возвращает категории и id категорий для офферов в фиде
     * https://yandex.ru/support/partnermarket/elements/categories.html
     */
    private fun getFeedCategories(offerCategories: Collection<String>): Map<String, FeedCategory> {
        return offerCategories.asSequence()
            .map { it.trim() }
            .distinct()
            .sorted()
            .mapIndexed { idx, category ->
                FeedCategory(idx + 1L, null, category)
            }
            .associateBy { it.name }
    }

    fun convertRetailOffersToYml(feedOffers: Collection<FeedOffer>): YmlCatalog {
        val retailOffers = feedOffers.filter { it.retailFeedOfferParams != null }

        val categories = getFeedCategories(
            retailOffers.map { it.retailFeedOfferParams.category ?: DEFAULT_CATEGORY }
        )

        val convertedOffers = retailOffers.map {
            val offerParams = it.retailFeedOfferParams

            FeedOfferData(
                id = it.id!!,
                available = it.isAvailable,
                url = it.href,
                price = it.currentPrice.toString(),
                oldPrice = it.oldPrice?.toString(),
                currencyId = it.currency.name,
                categoryId = categories[offerParams.category?.trim() ?: DEFAULT_CATEGORY]!!.id,
                pictures = it.images,
                name = it.label,
                description = it.description,
                vendor = offerParams.vendor,
                model = offerParams.model
            )
        }

        val currencies = getFeedCurrencies(retailOffers.map { it.currency })

        return YmlCatalog(
            date = currentDateTime(),
            shop = FeedShop(
                currencies = currencies,
                categories = FeedCategories(categories.values.toList()),
                offers = FeedOffersData(convertedOffers)
            )
        )
    }
}

private val context = JAXBContext.newInstance(YmlCatalog::class.java)

fun YmlCatalog.toByteArray(formattedOutput: Boolean = false): ByteArray {
    val marshaller = context.createMarshaller()
    marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, formattedOutput)

    val stringWriter = StringWriter()
    marshaller.marshal(this, stringWriter)
    return stringWriter.toString().toByteArray()
}
