package ru.yandex.direct.core.entity.feedoffer.service;

import java.util.List;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.feedoffer.converter.FeedOfferConverter;
import ru.yandex.direct.core.entity.feedoffer.model.FeedOffer;
import ru.yandex.direct.core.entity.feedoffer.validation.FeedOfferValidationService;
import ru.yandex.direct.core.entity.uac.grut.GrutContext;
import ru.yandex.direct.core.grut.api.GrutApiProperties;
import ru.yandex.direct.core.grut.api.OfferGrutApi;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class FeedOfferService {

    private final OfferGrutApi offerGrutApi;
    private final FeedOfferValidationService feedOfferValidationService;

    @Autowired
    public FeedOfferService(GrutContext grutContext, GrutApiProperties grutApiProperties,
                            FeedOfferValidationService feedOfferValidationService) {
        this.offerGrutApi = new OfferGrutApi(grutContext, grutApiProperties);
        this.feedOfferValidationService = feedOfferValidationService;
    }

    public List<FeedOffer> getOffersByClientId(ClientId clientId) {
        return mapList(offerGrutApi.selectOffersByClientId(clientId.asLong()), FeedOfferConverter::fromFeedOfferGrut);
    }

    public List<Long> getOfferIdsByClientId(ClientId clientId) {
        return mapList(getOffersByClientId(clientId), FeedOffer::getId);
    }

    public MassResult<Long> addOrUpdateFeedOffers(
            ClientId clientId,
            Long clientUid,
            Long operatorUid,
            List<FeedOffer> offers
    ) {
        ValidationResult<List<FeedOffer>, Defect> validationResult =
                feedOfferValidationService.validate(clientId, clientUid, operatorUid, offers);
        List<Long> offerIds = mapList(offers, FeedOffer::getId);

        if (validationResult.hasAnyErrors()) {
            return MassResult.brokenMassAction(offerIds, validationResult);
        }

        List<Long> offerIdsToDelete = getOfferIdsByClientId(clientId).stream()
                .filter(id -> !offerIds.contains(id))
                .collect(Collectors.toList());
        var isNewToOffers = StreamEx.of(offers)
                .map(FeedOfferConverter::toFeedOfferGrut)
                .partitioningBy(offer -> offer.getId() == null);

        offerGrutApi.createOffers(isNewToOffers.get(true));
        offerGrutApi.createOrUpdateOffers(isNewToOffers.get(false));
        offerGrutApi.deleteOffers(offerIdsToDelete);

        return MassResult.successfulMassAction(offerIds, validationResult);
    }
}
