package ru.yandex.direct.core.entity.feedoffer.validation;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.feedoffer.converter.FeedOfferConverter;
import ru.yandex.direct.core.entity.feedoffer.model.FeedOffer;
import ru.yandex.direct.core.entity.feedoffer.model.RetailFeedOfferParams;
import ru.yandex.direct.core.entity.uac.grut.GrutContext;
import ru.yandex.direct.core.grut.api.ClientGrutApi;
import ru.yandex.direct.core.grut.api.GrutApiProperties;
import ru.yandex.direct.core.grut.api.OfferGrutApi;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.Predicates;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Objects.nonNull;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstraints.charsAreAllowed;
import static ru.yandex.direct.core.entity.feedoffer.validation.FeedOfferDefects.clientNotExistInGrut;
import static ru.yandex.direct.core.entity.feedoffer.validation.FeedOfferDefects.hrefNotUnique;
import static ru.yandex.direct.core.entity.feedoffer.validation.FeedOfferDefects.imageCountIsTooLarge;
import static ru.yandex.direct.core.entity.feedoffer.validation.FeedOfferDefects.noImages;
import static ru.yandex.direct.core.entity.feedoffer.validation.FeedOfferDefects.noOffers;
import static ru.yandex.direct.core.entity.feedoffer.validation.FeedOfferDefects.offerCountIsTooLarge;
import static ru.yandex.direct.core.entity.feedoffer.validation.FeedOfferDefects.offerHasInvalidHref;
import static ru.yandex.direct.core.entity.feedoffer.validation.FeedOfferDefects.offerIdNotUnique;
import static ru.yandex.direct.core.entity.feedoffer.validation.FeedOfferDefects.oldPriceLessOrEqualThanCurrent;
import static ru.yandex.direct.core.entity.feedoffer.validation.FeedOfferDefects.unknownOfferId;
import static ru.yandex.direct.core.validation.defects.RightsDefects.noRights;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.unconditional;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThanOrEqualTo;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;

@Service
public class FeedOfferValidationService {

    private static final Integer MAX_OFFER_COUNT = 1000;
    private static final Integer MAX_LABEL_LENGTH = 56;
    private static final Integer MAX_DESCRIPTION_LENGTH = 81;
    private static final Integer MAX_IMAGE_COUNT = 5;
    private static final BigDecimal MIN_PRICE = BigDecimal.valueOf(0);
    private static final BigDecimal MAX_PRICE = BigDecimal.valueOf(10_000_000_000L);

    private final RbacService rbacService;
    private final ClientGrutApi clientGrutApi;
    private final OfferGrutApi offerGrutApi;

    @Autowired
    public FeedOfferValidationService(RbacService rbacService, GrutContext grutContext,
                                      GrutApiProperties grutApiProperties) {
        this.rbacService = rbacService;
        this.clientGrutApi = new ClientGrutApi(grutContext, grutApiProperties);
        this.offerGrutApi = new OfferGrutApi(grutContext, grutApiProperties);
    }

    public ValidationResult<List<FeedOffer>, Defect> validate(ClientId clientId, Long clientUid, Long operatorUid,
                                                              List<FeedOffer> offers) {
        ListValidationBuilder<FeedOffer, Defect> lvb = ListValidationBuilder.of(offers);
        boolean canWrite = rbacService.canWrite(operatorUid, clientUid);
        lvb.check(unconditional(noRights()), When.isFalse(canWrite))
                .check(existingClientInGrut(clientId), When.isValid());
        if (lvb.getResult().hasAnyErrors()) {
            return lvb.getResult();
        }

        List<FeedOffer> existingOffers = mapList(offerGrutApi.selectOffersByClientId(clientId.asLong()),
                FeedOfferConverter::fromFeedOfferGrut);
        Set<Long> existingIds = listToSet(existingOffers, FeedOffer::getId);

        Map<String, Long> hrefsToCount = StreamEx.of(offers)
                .map(FeedOffer::getHref)
                .nonNull()
                .groupingBy(identity(), Collectors.counting());

        Map<Long, Long> idsToCount = StreamEx.of(offers)
                .map(FeedOffer::getId)
                .nonNull()
                .groupingBy(identity(), Collectors.counting());

        return lvb
                .check(notEmptyCollection(), noOffers())
                .check(validOffersCount())
                .checkEachBy(validateFeedOffer(existingIds, hrefsToCount, idsToCount), When.isValid())
                .getResult();
    }

    private Constraint<List<FeedOffer>, Defect> existingClientInGrut(ClientId clientId) {
        return offers -> (nonNull(clientGrutApi.getClient(clientId.asLong()))) ? null : clientNotExistInGrut();
    }

    private Constraint<List<FeedOffer>, Defect> validOffersCount() {
        return fromPredicate(Predicates.listSize(0, MAX_OFFER_COUNT), offerCountIsTooLarge(MAX_OFFER_COUNT));
    }

    private Validator<FeedOffer, Defect> validateFeedOffer(Set<Long> existingIds, Map<String, Long> hrefsToCount,
                                                           Map<Long, Long> idsToCount) {
        return feedOffer -> {
            ModelItemValidationBuilder<FeedOffer> vb = ModelItemValidationBuilder.of(feedOffer);
            vb.item(FeedOffer.ID)
                    .check(isIdUnique(idsToCount), When.notNull())
                    .check(existingOfferId(existingIds), When.notNull());
            vb.item(FeedOffer.LABEL)
                    .check(notNull())
                    .check(notBlank(), When.isValid())
                    .check(maxStringLength(MAX_LABEL_LENGTH), When.isValid())
                    .check(charsAreAllowed(), When.isValid());
            vb.item(FeedOffer.DESCRIPTION)
                    .check(notNull())
                    .check(notBlank(), When.isValid())
                    .check(maxStringLength(MAX_DESCRIPTION_LENGTH), When.isValid())
                    .check(charsAreAllowed(), When.isValid());
            vb.item(FeedOffer.HREF)
                    .check(notNull())
                    .check(notBlank(), When.isValid())
                    .check(validHref(), offerHasInvalidHref(), When.isValid())
                    .check(isHrefUnique(hrefsToCount), When.isValid());
            vb.item(FeedOffer.IMAGES)
                    .check(notNull())
                    .check(notEmptyCollection(), noImages(), When.isValid())
                    .check(validImagesCount(), When.isValid());
            vb.item(FeedOffer.CURRENCY)
                    .check(notNull());
            vb.item(FeedOffer.CURRENT_PRICE)
                    .check(notNull())
                    .check(notLessThanOrEqualTo(MIN_PRICE), When.isValid())
                    .check(notGreaterThan(MAX_PRICE), When.isValid());
            vb.item(FeedOffer.OLD_PRICE)
                    .check(notLessThanOrEqualTo(MIN_PRICE), When.notNull())
                    .check(notGreaterThan(MAX_PRICE), When.notNull())
                    .check(notLessThanOrEqualTo(feedOffer.getCurrentPrice()),
                            oldPriceLessOrEqualThanCurrent(feedOffer.getCurrentPrice()), When.notNull());
            vb.item(FeedOffer.IS_AVAILABLE)
                    .check(notNull());
            vb.item(FeedOffer.RETAIL_FEED_OFFER_PARAMS)
                    .check(notNull())
                    .checkBy(validateRetailParams(), When.isValid());

            return vb.getResult();
        };
    }

    private Constraint<Long, Defect> isIdUnique(Map<Long, Long> idsToCount) {
        return id -> (idsToCount.get(id) == 1) ? null : offerIdNotUnique();
    }

    private Constraint<Long, Defect> existingOfferId(Set<Long> existingIds) {
        return id -> (existingIds.contains(id)) ? null : unknownOfferId();
    }

    private Constraint<String, Defect> isHrefUnique(Map<String, Long> hrefsToCount) {
        return href -> (hrefsToCount.get(href) == 1) ? null : hrefNotUnique();
    }

    private Constraint<List<String>, Defect> validImagesCount() {
        return fromPredicate(Predicates.listSize(0, MAX_IMAGE_COUNT), imageCountIsTooLarge(MAX_IMAGE_COUNT));
    }

    private Validator<RetailFeedOfferParams, Defect> validateRetailParams() {
        return params -> {
            ModelItemValidationBuilder<RetailFeedOfferParams> vb = ModelItemValidationBuilder.of(params);
            vb.item(RetailFeedOfferParams.CATEGORY)
                    .check(charsAreAllowed(), When.notNull());
            vb.item(RetailFeedOfferParams.VENDOR)
                    .check(charsAreAllowed(), When.notNull());
            vb.item(RetailFeedOfferParams.MODEL)
                    .check(charsAreAllowed(), When.notNull());

            return vb.getResult();
        };
    }
}
