package ru.yandex.direct.core.entity.forecast;

import java.math.BigInteger;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;

import org.jooq.Field;
import org.jooq.Record;
import org.jooq.SelectJoinStep;
import org.jooq.types.ULong;
import org.jooq.types.UNumber;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.forecast.model.CpaEstimate;
import ru.yandex.direct.core.entity.forecast.model.CpaEstimateAttributionModel;
import ru.yandex.direct.core.entity.forecast.model.CpaEstimateGoalType;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.core.entity.forecast.model.CpaEstimateAttributionModel.toSource;
import static ru.yandex.direct.dbschema.ppcdict.tables.CpaEstimates.CPA_ESTIMATES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.HashingUtils.getMd5HalfHashUtf8;

@Repository
public class CpaEstimatesRepository {

    private final DslContextProvider dslContextProvider;

    private final JooqMapperWithSupplier<CpaEstimate> mapper;
    private final Set<Field<?>> fieldsToRead;

    @Autowired
    public CpaEstimatesRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        this.mapper = JooqMapperWithSupplierBuilder.builder(CpaEstimate::new)
                .map(convertibleProperty(CpaEstimate.ATTRIBUTION_MODEL, CPA_ESTIMATES.ATTRIBUTION_MODEL,
                        CpaEstimateAttributionModel::fromSource,
                        CpaEstimateAttributionModel::toSource))
                .map(property(CpaEstimate.BUSINESS_CATEGORY, CPA_ESTIMATES.BUSINESS_CATEGORY))
                .map(convertibleProperty(CpaEstimate.BUSINESS_CATEGORY_HASH, CPA_ESTIMATES.BUSINESS_CATEGORY_HASH,
                        t -> ifNotNull(t, UNumber::toBigInteger), t -> ifNotNull(t, ULong::valueOf)))
                .map(property(CpaEstimate.REGION_ID, CPA_ESTIMATES.REGION_ID))
                .map(convertibleProperty(CpaEstimate.GOAL_TYPE, CPA_ESTIMATES.GOAL_TYPE,
                        CpaEstimateGoalType::fromSource,
                        CpaEstimateGoalType::toSource))
                .map(property(CpaEstimate.MEDIAN_CPA, CPA_ESTIMATES.MEDIAN_CPA))
                .map(property(CpaEstimate.MIN_CPA, CPA_ESTIMATES.MIN_CPA))
                .map(property(CpaEstimate.MAX_CPA, CPA_ESTIMATES.MAX_CPA))
                .build();
        this.fieldsToRead = mapper.getFieldsToRead();
    }

    public List<CpaEstimate> getCpaEstimates(@Nullable String businessCategory,
                                             @Nullable List<Long> regionIds,
                                             @Nullable CpaEstimateAttributionModel attributionModel) {
        var selectStep = dslContextProvider.ppcdict()
                .select(fieldsToRead)
                .from(CPA_ESTIMATES);

        applyFilters(selectStep, businessCategory, regionIds, attributionModel);

        return selectStep
                .fetch(mapper::fromDb);
    }

    private void applyFilters(SelectJoinStep<Record> selectStep,
                              @Nullable String businessCategory,
                              @Nullable List<Long> regionIds,
                              @Nullable CpaEstimateAttributionModel attributionModel) {
        if (regionIds != null) {
            selectStep
                    .where(CPA_ESTIMATES.REGION_ID.in(regionIds));
        }
        if (businessCategory != null) {
            BigInteger businessCategoryHash = getHash(businessCategory);

            selectStep
                    .where(CPA_ESTIMATES.BUSINESS_CATEGORY_HASH.eq(ULong.valueOf(businessCategoryHash)));
        }
        if (attributionModel != null) {
            selectStep
                    .where(CPA_ESTIMATES.ATTRIBUTION_MODEL.eq(toSource(attributionModel)));
        }
    }

    public void addOrUpdate(List<CpaEstimate> cpaEstimates) {
        fillHash(cpaEstimates);

        new InsertHelper<>(dslContextProvider.ppcdict(), CPA_ESTIMATES)
                .addAll(mapper, cpaEstimates)
                .onDuplicateKeyUpdate()
                .set(CPA_ESTIMATES.MEDIAN_CPA, MySQLDSL.values(CPA_ESTIMATES.MEDIAN_CPA))
                .set(CPA_ESTIMATES.MIN_CPA, MySQLDSL.values(CPA_ESTIMATES.MIN_CPA))
                .set(CPA_ESTIMATES.MAX_CPA, MySQLDSL.values(CPA_ESTIMATES.MAX_CPA))
                .executeIfRecordsAdded();
    }

    private void fillHash(List<CpaEstimate> cpaEstimates) {
        cpaEstimates.forEach(cpaEstimate ->
                cpaEstimate.withBusinessCategoryHash(getHash(cpaEstimate.getBusinessCategory())));
    }

    private BigInteger getHash(String businessCategory) {
        if (businessCategory == null) {
            return null;
        }

        return getMd5HalfHashUtf8(businessCategory);
    }
}
