package ru.yandex.direct.core.entity.freelancer.container;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.annotation.Nullable;

import static java.util.Arrays.asList;
import static java.util.Collections.unmodifiableList;

/**
 * Контейнер для фильтров, применяемых при отборе фрилансеров
 */
public class FreelancersQueryFilter {

    private final List<Long> freelancerIds;
    private final boolean onlySearchable;
    private final boolean skipDisabled;

    private FreelancersQueryFilter(Builder builder) {
        onlySearchable = builder.onlySearchable;
        freelancerIds =
                builder.freelancerIds == null ? null : unmodifiableList(new ArrayList<>(builder.freelancerIds));
        skipDisabled = builder.skipDisabled;
    }

    @Nullable
    public List<Long> getFreelancerIds() {
        return freelancerIds;
    }

    public boolean isOnlySearchable() {
        return onlySearchable;
    }

    public boolean isSkipDisabled() {
        return skipDisabled;
    }

    /**
     * {@link Builder} с фильтрами по умолчанию (не отдаёт выключенных и скрытых специалистов)
     */
    public static Builder activeFreelancers() {
        return new Builder();
    }

    /**
     * Все специалисты за исключением выключенных.
     * Специалисты, скрытые из списка, также будут возвращены.
     *
     * @see #activeFreelancers()
     */
    public static Builder enabledFreelancers() {
        return activeFreelancers()
                .withNonSearchable();
    }

    /**
     * Все специалисты, включая отключенных.
     * <p>
     * <b>Внимание:</b> данный фильтр не должен использоваться при обработке запросов внешних пользователей,
     * так как для них отключенные специалисты не существуют.
     *
     * @see #activeFreelancers()
     * @see #enabledFreelancers()
     */
    public static Builder allFreelancersIncludingDisabled() {
        return enabledFreelancers()
                .withDisabled();
    }

    public static class Builder {

        private Collection<Long> freelancerIds = null;
        private boolean onlySearchable = true;
        private boolean skipDisabled = true;

        private Builder() {
        }

        public Builder withFreelancerIds(Collection<Long> freelancerIds) {
            this.freelancerIds = new ArrayList<>(freelancerIds);
            return this;
        }

        public Builder withFreelancerIds(Long... freelancerIds) {
            return withFreelancerIds(asList(freelancerIds));
        }

        private Builder withNonSearchable() {
            this.onlySearchable = false;
            return this;
        }

        private Builder withDisabled() {
            this.skipDisabled = false;
            return this;
        }

        public FreelancersQueryFilter build() {
            return new FreelancersQueryFilter(this);
        }
    }
}
