package ru.yandex.direct.core.entity.freelancer.operation;

import java.util.Collection;
import java.util.List;

import ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter;
import ru.yandex.direct.core.entity.freelancer.model.Freelancer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerBase;
import ru.yandex.direct.core.entity.freelancer.repository.FreelancerRepository;
import ru.yandex.direct.core.entity.freelancer.service.validation.UpdateFreelancerValidationService;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter.allFreelancersIncludingDisabled;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Операция обновления параметров фрилансера (без карточки)
 */
public class FreelancerUpdateOperation extends SimpleAbstractUpdateOperation<FreelancerBase, Long> {
    private final FreelancerRepository freelancerRepository;
    private final UpdateFreelancerValidationService validationService;

    private final int shard;

    public FreelancerUpdateOperation(
            List<ModelChanges<FreelancerBase>> modelChanges,
            FreelancerRepository freelancerRepository,
            UpdateFreelancerValidationService validationService,
            int shard) {
        super(Applicability.FULL, modelChanges, id -> new FreelancerBase().withId(id));
        this.freelancerRepository = freelancerRepository;
        this.validationService = validationService;
        this.shard = shard;
    }

    @Override
    protected Collection<FreelancerBase> getModels(Collection<Long> ids) {
        // берём фильтр для всех специалистов, так как операция используется в том числе из внутреннего отчёта
        FreelancersQueryFilter filter = allFreelancersIncludingDisabled().withFreelancerIds(ids).build();
        List<Freelancer> freelancers = freelancerRepository.getByFilter(shard, filter);
        return mapList(freelancers, FreelancerBase.class::cast);
    }

    @Override
    protected ValidationResult<List<ModelChanges<FreelancerBase>>, Defect> validateModelChanges(
            List<ModelChanges<FreelancerBase>> modelChanges) {
        return validationService.preValidate(modelChanges);
    }

    @Override
    protected ValidationResult<List<FreelancerBase>, Defect> validateAppliedChanges(
            ValidationResult<List<FreelancerBase>, Defect> validationResult) {
        return validationService.validate(validationResult);
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<FreelancerBase>> applicableAppliedChanges) {
        freelancerRepository.updateFreelancer(shard, applicableAppliedChanges);

        return applicableAppliedChanges.stream()
                .map(AppliedChanges::getModel)
                .map(ModelWithId::getId)
                .collect(toList());
    }
}
