package ru.yandex.direct.core.entity.freelancer.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DatePart;
import org.jooq.Field;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.jooqmapper.OldJooqMapperBuilder;
import ru.yandex.direct.common.jooqmapper.OldJooqMapperWithSupplier;
import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.freelancer.model.ClientAvatar;
import ru.yandex.direct.core.entity.freelancer.model.ClientAvatarsHost;
import ru.yandex.direct.dbschema.ppc.tables.records.ClientsAvatarsRecord;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.dbutil.wrapper.ShardedDb;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.Collections.emptyList;
import static org.jooq.impl.DSL.currentLocalDateTime;
import static org.jooq.impl.DSL.localDateTimeSub;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.convertibleField;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.field;
import static ru.yandex.direct.common.jooqmapperex.FieldMapperFactoryEx.booleanField;
import static ru.yandex.direct.common.util.RepositoryUtils.FALSE;
import static ru.yandex.direct.common.util.RepositoryUtils.TRUE;
import static ru.yandex.direct.dbschema.ppc.tables.ClientsAvatars.CLIENTS_AVATARS;
import static ru.yandex.direct.dbschema.ppc.tables.FreelancersCard.FREELANCERS_CARD;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class ClientAvatarRepository {
    private final OldJooqMapperWithSupplier<ClientAvatar> clientAvatarMapper;
    private final DatabaseWrapperProvider databaseWrapperProvider;

    public ClientAvatarRepository(DatabaseWrapperProvider databaseWrapperProvider) {
        this.databaseWrapperProvider = databaseWrapperProvider;
        clientAvatarMapper = createMapper();
    }

    private static OldJooqMapperWithSupplier<ClientAvatar> createMapper() {
        return new OldJooqMapperBuilder<>(ClientAvatar::new)
                .map(field(CLIENTS_AVATARS.AVATAR_ID, ClientAvatar.ID))
                .map(field(CLIENTS_AVATARS.CLIENT_ID, ClientAvatar.CLIENT_ID))
                .map(field(CLIENTS_AVATARS.EXTERNAL_ID, ClientAvatar.EXTERNAL_ID))
                .map(convertibleField(CLIENTS_AVATARS.HOST_CONFIG_NAME, ClientAvatar.HOST)
                        .convertToDbBy(ClientAvatarsHost::toSource)
                        .convertFromDbBy(ClientAvatarsHost::fromSource)
                )
                .map(field(CLIENTS_AVATARS.CREATE_TIME, ClientAvatar.CREATE_TIME))
                .map(booleanField(CLIENTS_AVATARS.IS_DELETED, ClientAvatar.IS_DELETED))
                .build();
    }

    public List<Long> add(int shard, List<ClientAvatar> avatars) {
        InsertHelper<ClientsAvatarsRecord> insertHelper =
                new InsertHelper<>(databaseWrapperProvider.get(ShardedDb.PPC, shard).getDslContext(), CLIENTS_AVATARS);
        insertHelper.addAll(clientAvatarMapper, avatars);
        insertHelper.execute();
        return mapList(avatars, ClientAvatar::getId);
    }

    public List<ClientAvatar> get(int shard, Collection<Long> avatarIds) {
        if (avatarIds.isEmpty()) {
            return emptyList();
        }
        return databaseWrapperProvider.get(ShardedDb.PPC, shard).getDslContext()
                .select(clientAvatarMapper.getFieldsToRead())
                .from(CLIENTS_AVATARS)
                .where(CLIENTS_AVATARS.AVATAR_ID.in(avatarIds))
                .and(CLIENTS_AVATARS.IS_DELETED.eq(FALSE))
                .fetch(clientAvatarMapper::fromDb);
    }

    /**
     * Возвращает список аватарок, которые созданы более {@code expirationPeriod} секунд назад,
     * но так и не привязаны к карточке фрилансера.
     */
    public List<ClientAvatar> getUnused(int shard, ClientAvatarsHost clientAvatarsHost, int expirationPeriod) {
        Field<LocalDateTime> expirationTime = localDateTimeSub(
                currentLocalDateTime(), expirationPeriod, DatePart.SECOND);
        return databaseWrapperProvider.get(ShardedDb.PPC, shard).getDslContext()
                .select(clientAvatarMapper.getFieldsToRead())
                .from(CLIENTS_AVATARS)
                .leftJoin(FREELANCERS_CARD)
                .on(CLIENTS_AVATARS.AVATAR_ID.equal(FREELANCERS_CARD.AVATAR_ID))
                .where(CLIENTS_AVATARS.HOST_CONFIG_NAME.equal(ClientAvatarsHost.toSource(clientAvatarsHost)))
                .and(FREELANCERS_CARD.FREELANCERS_CARD_ID.isNull())
                .and(CLIENTS_AVATARS.CREATE_TIME.lessThan(expirationTime))
                .fetch(clientAvatarMapper::fromDb);
    }

    /**
     * Обновляет только значение поля IS_DELETED, т.к. по жизненному циклу аватарки на данный момент
     * другие поля не должны изменяться.
     *
     * @param shard          шард на котором лежит запись.
     * @param appliedChanges запрос на изменения аватарок.
     * @return количество изменённых аватарок.
     */
    public int update(int shard, Collection<AppliedChanges<ClientAvatar>> appliedChanges) {
        if (appliedChanges.isEmpty()) {
            return 0;
        }
        JooqUpdateBuilder<ClientsAvatarsRecord, ClientAvatar> updateBuilder =
                new JooqUpdateBuilder<>(CLIENTS_AVATARS.AVATAR_ID, appliedChanges);
        updateBuilder.processProperty(ClientAvatar.IS_DELETED, CLIENTS_AVATARS.IS_DELETED,
                RepositoryUtils::booleanToLong);
        return databaseWrapperProvider.get(ShardedDb.PPC, shard).getDslContext()
                .update(CLIENTS_AVATARS)
                .set(updateBuilder.getValues())
                .where(CLIENTS_AVATARS.AVATAR_ID.in(updateBuilder.getChangedIds()))
                .execute();
    }

    /**
     * Удаляет аватарки, у которых в БД стоит {@code is_deleted=0}
     */
    public int delete(int shard, Collection<Long> avatarIds) {
        if (avatarIds.isEmpty()) {
            return 0;
        }
        return databaseWrapperProvider.get(ShardedDb.PPC, shard).getDslContext()
                .delete(CLIENTS_AVATARS)
                .where(CLIENTS_AVATARS.AVATAR_ID.in(avatarIds))
                .and(CLIENTS_AVATARS.IS_DELETED.eq(TRUE))
                .execute();
    }
}
