package ru.yandex.direct.core.entity.freelancer.repository;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Record1;
import org.jooq.SelectHavingStep;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.jooqmapper.OldJooqMapperBuilder;
import ru.yandex.direct.common.jooqmapper.OldJooqMapperWithSupplier;
import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCard;
import ru.yandex.direct.core.entity.freelancer.model.FreelancersCardStatusModerate;
import ru.yandex.direct.dbschema.ppc.tables.records.FreelancersCardRecord;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static org.jooq.impl.DSL.max;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.convertibleField;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.field;
import static ru.yandex.direct.common.jooqmapperex.FieldMapperFactoryEx.booleanField;
import static ru.yandex.direct.common.util.RepositoryUtils.FALSE;
import static ru.yandex.direct.dbschema.ppc.Tables.FREELANCERS_CARD;
import static ru.yandex.direct.dbschema.ppc.enums.FreelancersCardStatusModerate.accepted;
import static ru.yandex.direct.dbschema.ppc.enums.FreelancersCardStatusModerate.draft;

@Repository
public class FreelancerCardRepository {
    private final DslContextProvider dslContextProvider;
    private final OldJooqMapperWithSupplier<FreelancerCard> freelancerCardMapper;
    private final ShardHelper shardHelper;

    public FreelancerCardRepository(DslContextProvider dslContextProvider,
                                    ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        freelancerCardMapper = createCardMapper();
    }

    private OldJooqMapperWithSupplier<FreelancerCard> createCardMapper() {
        return new OldJooqMapperBuilder<>(FreelancerCard::new)
                .map(field(FREELANCERS_CARD.FREELANCERS_CARD_ID, FreelancerCard.ID))
                .map(field(FREELANCERS_CARD.CLIENT_ID, FreelancerCard.FREELANCER_ID))
                .map(field(FREELANCERS_CARD.AVATAR_ID, FreelancerCard.AVATAR_ID))
                .map(field(FREELANCERS_CARD.BRIEF, FreelancerCard.BRIEF_INFO))
                .map(convertibleField(FREELANCERS_CARD.CONTACTS, FreelancerCard.CONTACTS)
                        .convertToDbBy(FreelancerMappings::contactsToJson)
                        .convertFromDbBy(FreelancerMappings::contactsFromJson))
                .map(convertibleField(FREELANCERS_CARD.STATUS_MODERATE, FreelancerCard.STATUS_MODERATE)
                        .convertToDbBy(FreelancersCardStatusModerate::toSource)
                        .convertFromDbBy(FreelancersCardStatusModerate::fromSource))
                .map(convertibleField(FREELANCERS_CARD.DECLINE_REASON, FreelancerCard.DECLINE_REASON)
                        .convertToDbBy(FreelancerCardMappings::declineReasonsToDb)
                        .convertFromDbBy(FreelancerCardMappings::declineReasonsFromDb))
                .map(booleanField(FREELANCERS_CARD.IS_ARCHIVED, FreelancerCard.IS_ARCHIVED))
                .build();
    }

    /**
     * Возвращает карточки по заданным ID вне зависимости от статусов.
     */
    public List<FreelancerCard> getFreelancerCards(int shard, Collection<Long> freelancerCardIds) {
        if (freelancerCardIds.isEmpty()) {
            return Collections.emptyList();
        }
        return dslContextProvider.ppc(shard)
                .select(freelancerCardMapper.getFieldsToRead())
                .from(FREELANCERS_CARD)
                .where(FREELANCERS_CARD.FREELANCERS_CARD_ID.in(freelancerCardIds))
                .fetch(freelancerCardMapper::fromDb);
    }

    /**
     * По идентификаторам фрилансеров возвращает их последние (новейшие) карточки, независимо от их статуса модераци или архивности.
     */
    @QueryWithoutIndex("Индекса нет по внешнему запросу, по подзапросу есть, выборка небольшая")
    public List<FreelancerCard> getNewestFreelancerCard(int shard, List<Long> freelancerIds) {
        SelectHavingStep<Record1<Long>> maxIdRecords = dslContextProvider.ppc(shard)
                .select(max(FREELANCERS_CARD.FREELANCERS_CARD_ID))
                .from(FREELANCERS_CARD)
                .where(FREELANCERS_CARD.CLIENT_ID.in(freelancerIds))
                .groupBy(FREELANCERS_CARD.CLIENT_ID);
        return dslContextProvider.ppc(shard)
                .select(freelancerCardMapper.getFieldsToRead())
                .from(FREELANCERS_CARD)
                .where(FREELANCERS_CARD.FREELANCERS_CARD_ID.in(maxIdRecords))
                .fetch(freelancerCardMapper::fromDb);
    }

    /**
     * По идентификаторам фрилансеров возвращает самые свежие из не архивных и одобренных модерацией карточек.
     */
    public Map<Long, FreelancerCard> getAcceptedCardsByFreelancerIds(int shard, Collection<Long> freelancerIds) {
        if (freelancerIds.isEmpty()) {
            return emptyMap();
        }
        List<FreelancerCard> freelancerCards = dslContextProvider.ppc(shard)
                .select(freelancerCardMapper.getFieldsToRead())
                .from(FREELANCERS_CARD)
                .where(FREELANCERS_CARD.CLIENT_ID.in(freelancerIds))
                .and(FREELANCERS_CARD.STATUS_MODERATE.eq(accepted))
                .and(FREELANCERS_CARD.IS_ARCHIVED.eq(FALSE))
                .fetch(freelancerCardMapper::fromDb);
        Map<Long, Optional<FreelancerCard>> grouping = StreamEx.of(freelancerCards)
                .mapToEntry(FreelancerCard::getFreelancerId, Function.identity())
                .grouping(Collectors.maxBy(Comparator.comparing(FreelancerCard::getId)));
        return EntryStream.of(grouping)
                .filterValues(Optional::isPresent)
                .mapValues(Optional::get)
                .toMap();
    }

    /**
     * Возвращает все карточки указанного фрилансера, кроме архивных.
     */
    public List<FreelancerCard> getNotArchivedFreelancerCards(int shard, Long freelancerId) {
        return dslContextProvider.ppc(shard)
                .select(freelancerCardMapper.getFieldsToRead())
                .from(FREELANCERS_CARD)
                .where(FREELANCERS_CARD.CLIENT_ID.eq(freelancerId))
                .and(FREELANCERS_CARD.IS_ARCHIVED.eq(FALSE))
                .fetch(freelancerCardMapper::fromDb);
    }

    /**
     * Возвращает последнюю (самую свежую) карточку указанного фрилансера,
     * которая была одобрена модерацией, независимо от архивности.
     */
    public Optional<FreelancerCard> getNewestAcceptedFreelancerCard(int shard, Long freelancerId) {
        List<FreelancerCard> freelancerCards = dslContextProvider.ppc(shard)
                .select(freelancerCardMapper.getFieldsToRead())
                .from(FREELANCERS_CARD)
                .where(FREELANCERS_CARD.CLIENT_ID.eq(freelancerId))
                .and(FREELANCERS_CARD.STATUS_MODERATE.eq(accepted))
                .fetch(freelancerCardMapper::fromDb);
        return StreamEx.of(freelancerCards).maxBy(FreelancerCard::getId);
    }

    /**
     * Возвращает все карточки ожидающие модерации.
     */
    @QueryWithoutIndex("Таблица небольшая, используется только в jobs")
    public List<FreelancerCard> getDraftFreelancerCards(int shard) {
        return dslContextProvider.ppc(shard)
                .select(freelancerCardMapper.getFieldsToRead())
                .from(FREELANCERS_CARD)
                .where(FREELANCERS_CARD.STATUS_MODERATE.eq(draft))
                .and(FREELANCERS_CARD.IS_ARCHIVED.eq(FALSE))
                .fetch(freelancerCardMapper::fromDb);
    }

    /**
     * Добавляет карточки фрилансеров.
     */
    public List<Long> addFreelancerCards(int shard, Collection<FreelancerCard> freelancerCards) {
        if (freelancerCards.isEmpty()) {
            return emptyList();
        }
        generateIds(freelancerCards);
        InsertHelper<FreelancersCardRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), FREELANCERS_CARD);
        for (FreelancerCard card : freelancerCards) {
            insertHelper
                    .add(freelancerCardMapper, card)
                    .newRecord();
        }
        insertHelper.execute();
        return StreamEx.of(freelancerCards)
                .map(FreelancerCard::getId)
                .toList();
    }

    private void generateIds(Collection<FreelancerCard> freelancerCards) {
        List<Long> ids = shardHelper.generateFreelancersCardIds(freelancerCards.size());
        StreamEx.of(freelancerCards).zipWith(ids.stream())
                .forKeyValue(FreelancerCard::setId);
    }

    /**
     * Меняет только статусы модерации и/или флаг архивности карточек фрилансеров.
     * Предполагается, что остальные поля карточек иммутабельны.
     */
    public void updateFreelancerCards(int shard, Collection<AppliedChanges<FreelancerCard>> appliedChanges) {
        JooqUpdateBuilder<ru.yandex.direct.dbschema.ppc.tables.records.FreelancersCardRecord, FreelancerCard>
                updateBuilder = new JooqUpdateBuilder<>(FREELANCERS_CARD.FREELANCERS_CARD_ID, appliedChanges);
        updateBuilder.processProperty(FreelancerCard.STATUS_MODERATE, FREELANCERS_CARD.STATUS_MODERATE,
                FreelancersCardStatusModerate::toSource);
        updateBuilder.processProperty(FreelancerCard.DECLINE_REASON, FREELANCERS_CARD.DECLINE_REASON,
                FreelancerCardMappings::declineReasonsToDb);
        updateBuilder.processProperty(FreelancerCard.IS_ARCHIVED, FREELANCERS_CARD.IS_ARCHIVED,
                RepositoryUtils::booleanToLong);
        dslContextProvider.ppc(shard)
                .update(FREELANCERS_CARD)
                .set(updateBuilder.getValues())
                .where(FREELANCERS_CARD.FREELANCERS_CARD_ID.in(updateBuilder.getChangedIds()))
                .execute();
    }
}
