package ru.yandex.direct.core.entity.freelancer.repository;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Condition;
import org.jooq.impl.DSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.jooqmapper.OldJooqMapperBuilder;
import ru.yandex.direct.common.jooqmapper.OldJooqMapperWithSupplier;
import ru.yandex.direct.core.entity.freelancer.container.FreelancerProjectQueryContainer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerProject;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerProjectStatus;
import ru.yandex.direct.dbschema.ppc.tables.records.FreelancersProjectsRecord;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;
import ru.yandex.direct.dbutil.wrapper.ShardedDb;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.convertibleField;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.field;
import static ru.yandex.direct.dbschema.ppc.tables.FreelancersProjects.FREELANCERS_PROJECTS;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class FreelancerProjectRepository {
    private final DatabaseWrapperProvider databaseWrapperProvider;
    private final OldJooqMapperWithSupplier<FreelancerProject> freelancerProjectMapper;

    public FreelancerProjectRepository(DatabaseWrapperProvider databaseWrapperProvider) {
        this.databaseWrapperProvider = databaseWrapperProvider;
        freelancerProjectMapper = createMapper();
    }

    private static OldJooqMapperWithSupplier<FreelancerProject> createMapper() {
        return new OldJooqMapperBuilder<>(FreelancerProject::new)
                .map(field(FREELANCERS_PROJECTS.PROJECT_ID, FreelancerProject.ID))
                .map(field(FREELANCERS_PROJECTS.FREELANCER_CLIENT_ID, FreelancerProject.FREELANCER_ID))
                .map(field(FREELANCERS_PROJECTS.CUSTOMER_CLIENT_ID, FreelancerProject.CLIENT_ID))
                .map(field(FREELANCERS_PROJECTS.CREATED_TIME, FreelancerProject.CREATED_TIME))
                .map(field(FREELANCERS_PROJECTS.STARTED_TIME, FreelancerProject.STARTED_TIME))
                .map(field(FREELANCERS_PROJECTS.UPDATED_TIME, FreelancerProject.UPDATED_TIME))
                .map(convertibleField(FREELANCERS_PROJECTS.STATUS, FreelancerProject.STATUS)
                        .convertToDbBy(FreelancerProjectStatus::toSource)
                        .convertFromDbBy(FreelancerProjectStatus::fromSource)
                )
                .build();
    }

    public List<FreelancerProject> get(int shard, FreelancerProjectQueryContainer queryContainer,
                                       LimitOffset limitOffset) {
        checkState(queryContainer.isEmpty(), "At least one of parameters must be specified");
        Condition condition = DSL.trueCondition();
        if (queryContainer.hasFreelancerIds()) {
            condition = condition.and(FREELANCERS_PROJECTS.FREELANCER_CLIENT_ID.in(queryContainer.getFreelancerIds()));
        }
        if (queryContainer.hasClientIds()) {
            condition = condition.and(FREELANCERS_PROJECTS.CUSTOMER_CLIENT_ID.in(queryContainer.getClientIds()));
        }
        if (queryContainer.hasProjectIds()) {
            condition = condition.and(FREELANCERS_PROJECTS.PROJECT_ID.in(queryContainer.getProjectIds()));
        }
        if (queryContainer.hasStatuses()) {
            condition = condition.and(FREELANCERS_PROJECTS.STATUS
                    .in(mapList(queryContainer.getStatuses(), FreelancerProjectStatus::toSource)));
        }
        if (queryContainer.getStarted() != null) {
            if (queryContainer.getStarted()) {
                condition = condition.and(FREELANCERS_PROJECTS.STARTED_TIME.isNotNull());
            } else {
                condition = condition.and(FREELANCERS_PROJECTS.STARTED_TIME.isNull());
            }
        }
        return databaseWrapperProvider.get(ShardedDb.PPC, shard).getDslContext()
                .select(freelancerProjectMapper.getFieldsToRead())
                .from(FREELANCERS_PROJECTS)
                .where(condition)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch(freelancerProjectMapper::fromDb);
    }

    public List<Long> add(int shard, List<FreelancerProject> projects) {
        InsertHelper<FreelancersProjectsRecord> bannersInsertHelper =
                new InsertHelper<>(databaseWrapperProvider.get(ShardedDb.PPC, shard).getDslContext(),
                        FREELANCERS_PROJECTS);
        bannersInsertHelper.addAll(freelancerProjectMapper, projects);
        return bannersInsertHelper.executeIfRecordsAddedAndReturn(FREELANCERS_PROJECTS.PROJECT_ID);
    }

    public int update(int shard, Collection<AppliedChanges<FreelancerProject>> appliedChanges) {
        JooqUpdateBuilder<FreelancersProjectsRecord, FreelancerProject> ub = createUpdateBuilder(appliedChanges);
        return databaseWrapperProvider.get(ShardedDb.PPC, shard).getDslContext()
                .update(FREELANCERS_PROJECTS)
                .set(ub.getValues())
                .where(FREELANCERS_PROJECTS.PROJECT_ID.in(ub.getChangedIds()))
                .execute();
    }

    private JooqUpdateBuilder<FreelancersProjectsRecord, FreelancerProject> createUpdateBuilder(
            Collection<AppliedChanges<FreelancerProject>> appliedChanges) {
        JooqUpdateBuilder<FreelancersProjectsRecord, FreelancerProject> updateBuilder =
                new JooqUpdateBuilder<>(FREELANCERS_PROJECTS.PROJECT_ID, appliedChanges);
        updateBuilder.processProperty(FreelancerProject.STARTED_TIME, FREELANCERS_PROJECTS.STARTED_TIME);
        updateBuilder.processProperty(FreelancerProject.UPDATED_TIME, FREELANCERS_PROJECTS.UPDATED_TIME);
        updateBuilder.processProperty(FreelancerProject.STATUS, FREELANCERS_PROJECTS.STATUS,
                FreelancerProjectStatus::toSource);
        return updateBuilder;
    }
}
