package ru.yandex.direct.core.entity.freelancer.repository;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import com.google.common.collect.Iterables;
import one.util.streamex.StreamEx;
import org.jooq.Record;
import org.jooq.SelectConditionStep;
import org.jooq.impl.DSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.jooqmapper.OldJooqMapperBuilder;
import ru.yandex.direct.common.jooqmapper.OldJooqMapperWithSupplier;
import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter;
import ru.yandex.direct.core.entity.freelancer.model.Freelancer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerBase;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCard;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerStatus;
import ru.yandex.direct.dbschema.ppc.enums.FreelancersIsSearchable;
import ru.yandex.direct.dbschema.ppc.tables.records.FreelancersRecord;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.convertibleField;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.field;
import static ru.yandex.direct.common.util.RepositoryUtils.FALSE;
import static ru.yandex.direct.core.entity.freelancer.model.Freelancer.fromBaseModel;
import static ru.yandex.direct.dbschema.ppc.Tables.FREELANCERS;

@Repository
public class FreelancerRepository {

    private final DslContextProvider dslContextProvider;
    private final FreelancerCardRepository freelancerCardRepository;
    private final OldJooqMapperWithSupplier<FreelancerBase> freelancerMapper;

    public FreelancerRepository(DslContextProvider dslContextProvider,
                                FreelancerCardRepository freelancerCardRepository) {
        this.dslContextProvider = dslContextProvider;
        this.freelancerCardRepository = freelancerCardRepository;
        freelancerMapper = createMapper();
    }

    private OldJooqMapperWithSupplier<FreelancerBase> createMapper() {
        return new OldJooqMapperBuilder<>(FreelancerBase::new)
                .map(field(FREELANCERS.CLIENT_ID, Freelancer.FREELANCER_ID))
                .map(field(FREELANCERS.FIRST_NAME, Freelancer.FIRST_NAME))
                .map(field(FREELANCERS.SECOND_NAME, Freelancer.SECOND_NAME))
                .map(field(FREELANCERS.CERT_LOGIN, Freelancer.CERT_LOGIN))
                .map(field(FREELANCERS.REGION_ID, Freelancer.REGION_ID))
                .map(convertibleField(FREELANCERS.CERTIFICATES, Freelancer.CERTIFICATES)
                        .convertToDbBy(FreelancerMappings::certificatesToJson)
                        .convertFromDbBy(FreelancerMappings::certificatesFromJson))
                .map(convertibleField(FREELANCERS.FREELANCER_STATUS, Freelancer.STATUS)
                        .convertToDbBy(FreelancerStatus::toSource)
                        .convertFromDbBy(FreelancerStatus::fromSource)
                        .withDatabaseDefault())
                .map(convertibleField(FREELANCERS.IS_SEARCHABLE, Freelancer.IS_SEARCHABLE)
                        .convertToDbBy(FreelancerMappings::searchableToDb)
                        .convertFromDbBy(FreelancerMappings::searchableFromDb)
                        .withDatabaseDefault())
                .map(convertibleField(FREELANCERS.IS_DISABLED, Freelancer.IS_DISABLED)
                        .convertToDbBy(RepositoryUtils::booleanToLong)
                        .convertFromDbBy(RepositoryUtils::booleanFromLong)
                        .withDatabaseDefault())
                .map(convertibleField(FREELANCERS.RATING, Freelancer.RATING)
                        .convertToDbBy(FreelancerMappings::ratingToDb)
                        .convertFromDbBy(FreelancerMappings::ratingFromDb)
                        .withDatabaseDefault())
                .map(field(FREELANCERS.FEEDBACK_COUNT, Freelancer.FEEDBACK_COUNT))
                .map(field(FREELANCERS.ADV_QUALITY_RATING, Freelancer.ADV_QUALITY_RATING))
                .map(field(FREELANCERS.ADV_QUALITY_RANK, Freelancer.ADV_QUALITY_RANK))
                .build();
    }


    /**
     * Возвращает ClientId всех активных фрилансеров.
     */
    @QueryWithoutIndex("Выборка всех доступных фрилансеров")
    public List<ClientId> getAllActiveId(int shard) {
        return dslContextProvider.ppc(shard)
                .select(FREELANCERS.CLIENT_ID)
                .from(FREELANCERS)
                .where(FREELANCERS.IS_DISABLED.eq(FALSE))
                .fetch(r -> ClientId.fromLong(r.get(FREELANCERS.CLIENT_ID)));
    }

    /**
     * Получить всех активных фрилансеров.
     */
    public List<Freelancer> getAllEnabledFreelancers(int shard) {
        return getByFilter(shard, FreelancersQueryFilter.enabledFreelancers().build());
    }

    /**
     * Получение фрилансеров по списку ID.
     * Эквивалентно вызову {@link #getByFilter(int, FreelancersQueryFilter)} с фильтром, включающим {@code ids}
     */
    public List<Freelancer> getByIds(int shard, Collection<Long> ids) {
        if (ids.isEmpty()) {
            return Collections.emptyList();
        }
        List<FreelancerBase> freelancerBases = getBaseFreelancers(shard, ids);
        return getFreelancersByBases(shard, freelancerBases);
    }

    private List<FreelancerBase> getBaseFreelancers(int shard, Collection<Long> ids) {
        return dslContextProvider.ppc(shard)
                .select(freelancerMapper.getFieldsToRead())
                .from(FREELANCERS)
                .where(FREELANCERS.CLIENT_ID.in(ids))
                .and(FREELANCERS.IS_DISABLED.eq(FALSE))
                .fetch(freelancerMapper::fromDb);
    }

    /**
     * Возвращает фрилансера с последнней (новейшей) сохранённой карточкой, независимо от её статуса модерации.
     */
    public Freelancer getFreelancerWithNewestCard(int shard, Long freelancerId) {
        List<FreelancerBase> baseFreelancers = getBaseFreelancers(shard, singletonList(freelancerId));
        if (baseFreelancers.isEmpty()) {
            return null;
        }
        FreelancerBase freelancerBase = baseFreelancers.get(0);
        Freelancer freelancer = fromBaseModel(freelancerBase);
        List<FreelancerCard> newestFreelancerCards = freelancerCardRepository
                .getNewestFreelancerCard(shard, singletonList(freelancerId));
        FreelancerCard freelancerCard = Iterables.getFirst(newestFreelancerCards, null);
        return freelancer.withCard(freelancerCard);
    }

    /**
     * Получение списка фрилансеров, удовлетворяющих заданному фильтру
     */
    @QueryWithoutIndex("Потенциальный full scan, но метод уже так написан")
    public List<Freelancer> getByFilter(int shard, FreelancersQueryFilter filter) {
        SelectConditionStep<Record> step = dslContextProvider.ppc(shard)
                .select(freelancerMapper.getFieldsToRead())
                .from(FREELANCERS)
                .where(DSL.trueCondition());
        if (filter.getFreelancerIds() != null) {
            step = step.and(FREELANCERS.CLIENT_ID.in(filter.getFreelancerIds()));
        }
        if (filter.isOnlySearchable()) {
            step = step.and(FREELANCERS.IS_SEARCHABLE.eq(FreelancersIsSearchable.Yes));
        }
        if (filter.isSkipDisabled()) {
            step = step.and(FREELANCERS.IS_DISABLED.eq(FALSE));
        }
        List<FreelancerBase> freelancerBases = step.fetch(freelancerMapper::fromDb);
        return getFreelancersByBases(shard, freelancerBases);
    }

    private List<Freelancer> getFreelancersByBases(int shard, List<FreelancerBase> freelancerBases) {
        List<Long> freelancerIds = StreamEx.of(freelancerBases)
                .map(FreelancerBase::getFreelancerId)
                .toList();
        Map<Long, FreelancerCard> freelancerCardByClientIds =
                freelancerCardRepository.getAcceptedCardsByFreelancerIds(shard, freelancerIds);
        return StreamEx.of(freelancerBases)
                .map(l -> fromBaseModel(l)
                        .withCard(freelancerCardByClientIds.get(l.getFreelancerId())))
                .toList();
    }

    /**
     * Добавляет {@code freelancers} в указанный шард
     */
    public void addFreelancers(int shard, Collection<Freelancer> freelancers) {
        if (freelancers.isEmpty()) {
            return;
        }
        List<FreelancerCard> freelancerCards = StreamEx.of(freelancers)
                .map(Freelancer::getCard)
                .filter(Objects::nonNull)
                .toList();
        addFreelancersToTable(shard, freelancers);
        if (freelancerCards.isEmpty()) {
            return;
        }
        freelancerCardRepository.addFreelancerCards(shard, freelancerCards);
    }

    private void addFreelancersToTable(int shard, Collection<? extends FreelancerBase> freelancers) {
        InsertHelper<FreelancersRecord> insertHelper
                = new InsertHelper<>(dslContextProvider.ppc(shard), FREELANCERS);
        for (FreelancerBase freelancer : freelancers) {
            insertHelper
                    .add(freelancerMapper, freelancer)
                    .newRecord();
        }
        insertHelper.execute();
    }

    public void updateFreelancer(int shard, List<AppliedChanges<FreelancerBase>> changes) {
        JooqUpdateBuilder<FreelancersRecord, FreelancerBase> ub =
                new JooqUpdateBuilder<>(FREELANCERS.CLIENT_ID, changes);

        ub.processProperty(FreelancerBase.IS_SEARCHABLE, FREELANCERS.IS_SEARCHABLE, FreelancerMappings::searchableToDb);
        ub.processProperty(FreelancerBase.IS_DISABLED, FREELANCERS.IS_DISABLED, RepositoryUtils::booleanToLong);
        ub.processProperty(FreelancerBase.RATING, FREELANCERS.RATING, FreelancerMappings::ratingToDb);
        ub.processProperty(FreelancerBase.FIRST_NAME, FREELANCERS.FIRST_NAME);
        ub.processProperty(FreelancerBase.SECOND_NAME, FREELANCERS.SECOND_NAME);
        ub.processProperty(FreelancerBase.CERT_LOGIN, FREELANCERS.CERT_LOGIN);
        ub.processProperty(FreelancerBase.REGION_ID, FREELANCERS.REGION_ID);
        ub.processProperty(FreelancerBase.FEEDBACK_COUNT, FREELANCERS.FEEDBACK_COUNT);
        ub.processProperty(FreelancerBase.STATUS, FREELANCERS.FREELANCER_STATUS, FreelancerStatus::toSource);
        ub.processProperty(FreelancerBase.CERTIFICATES, FREELANCERS.CERTIFICATES,
                FreelancerMappings::certificatesToJson);
        ub.processProperty(FreelancerBase.ADV_QUALITY_RATING, FREELANCERS.ADV_QUALITY_RATING);
        ub.processProperty(FreelancerBase.ADV_QUALITY_RANK, FREELANCERS.ADV_QUALITY_RANK);

        dslContextProvider.ppc(shard)
                .update(FREELANCERS)
                .set(ub.getValues())
                .where(FREELANCERS.CLIENT_ID.in(ub.getChangedIds()))
                .execute();
    }
}
