package ru.yandex.direct.core.entity.freelancer.repository;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.jooq.util.mysql.MySQLDSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.jooqmapper.OldJooqMapperBuilder;
import ru.yandex.direct.common.jooqmapper.OldJooqMapperWithSupplier;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerSkillOffer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerSkillOfferDuration;
import ru.yandex.direct.dbschema.ppc.tables.records.FreelancersSkillsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.convertibleField;
import static ru.yandex.direct.common.jooqmapper.FieldMapperFactory.field;
import static ru.yandex.direct.dbschema.ppc.Tables.FREELANCERS_SKILLS;

@Repository
public class FreelancerSkillsRepository {

    private final DslContextProvider dslContextProvider;
    private final OldJooqMapperWithSupplier<FreelancerSkillOffer> freelancerSkillMapper;

    public FreelancerSkillsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        freelancerSkillMapper = createMapper();
    }

    private OldJooqMapperWithSupplier<FreelancerSkillOffer> createMapper() {
        return new OldJooqMapperBuilder<>(FreelancerSkillOffer::new)
                .map(field(FREELANCERS_SKILLS.CLIENT_ID, FreelancerSkillOffer.FREELANCER_ID))
                .map(field(FREELANCERS_SKILLS.SKILL_ID, FreelancerSkillOffer.SKILL_ID))
                .map(convertibleField(FREELANCERS_SKILLS.DURATION, FreelancerSkillOffer.DURATION)
                        .convertFromDbBy(FreelancerSkillOfferDuration::fromDb)
                        .convertToDbBy(FreelancerSkillOfferDuration::toDb))
                .map(field(FREELANCERS_SKILLS.PRICE, FreelancerSkillOffer.PRICE))
                .build();
    }

    public List<FreelancerSkillOffer> getOffers(int shard, Collection<Long> freelancerIds) {
        if (freelancerIds.isEmpty()) {
            return Collections.emptyList();
        }
        return dslContextProvider.ppc(shard)
                .select(freelancerSkillMapper.getFieldsToRead())
                .from(FREELANCERS_SKILLS)
                .where(FREELANCERS_SKILLS.CLIENT_ID.in(freelancerIds))
                .fetch(freelancerSkillMapper::fromDb);
    }

    public void insertOrUpdate(int shard, Collection<? extends FreelancerSkillOffer> freelancerSkillOffers) {
        if (freelancerSkillOffers.isEmpty()) {
            return;
        }
        InsertHelper<FreelancersSkillsRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), FREELANCERS_SKILLS);
        for (FreelancerSkillOffer offer : freelancerSkillOffers) {
            insertHelper
                    .add(freelancerSkillMapper, offer)
                    .newRecord();
        }
        insertHelper.onDuplicateKeyUpdate()
                .set(FREELANCERS_SKILLS.CLIENT_ID, MySQLDSL.values(FREELANCERS_SKILLS.CLIENT_ID))
                .set(FREELANCERS_SKILLS.SKILL_ID, MySQLDSL.values(FREELANCERS_SKILLS.SKILL_ID))
                .set(FREELANCERS_SKILLS.DURATION, MySQLDSL.values(FREELANCERS_SKILLS.DURATION))
                .set(FREELANCERS_SKILLS.PRICE, MySQLDSL.values(FREELANCERS_SKILLS.PRICE))
                .execute();
    }

    public int delete(int shard, ClientId clientId, List<Long> skillOfferIds) {
        if (skillOfferIds.isEmpty()) {
            return 0;
        }
        return dslContextProvider.ppc(shard).deleteFrom(FREELANCERS_SKILLS)
                .where(FREELANCERS_SKILLS.CLIENT_ID.eq(clientId.asLong())
                        .and(FREELANCERS_SKILLS.SKILL_ID.in(skillOfferIds)))
                .execute();
    }
}
