package ru.yandex.direct.core.entity.freelancer.service;


import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalLong;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.freelancer.model.FreelancerCard;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCardModeration;
import ru.yandex.direct.core.entity.freelancer.model.FreelancersCardStatusModerate;
import ru.yandex.direct.core.entity.freelancer.operation.FreelancerCardAddOperation;
import ru.yandex.direct.core.entity.freelancer.repository.FreelancerCardRepository;
import ru.yandex.direct.core.entity.freelancer.service.utils.FreelancerCardUtils;
import ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerCardValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.freelancer.model.FreelancersCardStatusModerate.ACCEPTED;
import static ru.yandex.direct.core.entity.freelancer.model.FreelancersCardStatusModerate.DECLINED;
import static ru.yandex.direct.core.entity.freelancer.model.FreelancersCardStatusModerate.DRAFT;

/**
 * Сервис для модификации карточки фрилансеров.
 * <p>
 * Отделён от {@link FreelancerService}, так как зависит от validation service'ов, которые зависят от {@link FreelancerService}
 */
@Service

public class FreelancerCardService {

    private final ShardHelper shardHelper;
    private final FreelancerCardRepository freelancerCardRepository;
    private final FreelancerCardValidationService validationService;

    public FreelancerCardService(
            ShardHelper shardHelper,
            FreelancerCardRepository freelancerCardRepository,
            FreelancerCardValidationService freelancerCardValidationService) {
        this.validationService = freelancerCardValidationService;
        this.freelancerCardRepository = freelancerCardRepository;
        this.shardHelper = shardHelper;
    }


    /**
     * Сохраняет новую карточку фрилансера.
     * Если значения полей avatarId, briefInfo и siteUrl окажутся идентичны текущей промодерированной карточке
     * фрилансера (и если она у него вообще есть), то новая карточка будет иметь статус модерации ACCEPTED.
     * Во всех остальных случаях у новой созданной карточки будет статус модерации DRAFT.
     */
    public Result<Long> addChangedFreelancerCard(ClientId clientId, FreelancerCard cardChanged) {
        FreelancerCardAddOperation freelancerCardAddOperation = getChangedCardAddOperation(clientId, cardChanged);
        Result<Long> longResult = freelancerCardAddOperation.prepareAndApply().get(0);
        Long freelancerId = clientId.asLong();
        archiveFreelancerCards(freelancerId);
        return longResult;
    }

    /**
     * Операция на валидацию и сохранение новой карточки фрилансера с переданными изменениями.
     * Если значения полей avatarId, briefInfo и siteUrl окажутся идентичны текущей промодерированной карточке
     * фрилансера (и если она у него вообще есть), то новая карточка будет иметь статус модерации ACCEPTED.
     * Во всех остальных случаях у новой созданной карточки будет статус модерации DRAFT.
     */
    public FreelancerCardAddOperation getChangedCardAddOperation(ClientId clientId, FreelancerCard cardChanges) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        long freelancerId = clientId.asLong();
        List<FreelancerCard> newestFreelancerCards = freelancerCardRepository
                .getNewestFreelancerCard(shard, singletonList(freelancerId));
        FreelancerCard srcCard = newestFreelancerCards.isEmpty() ? new FreelancerCard() : newestFreelancerCards.get(0);
        FreelancerCard newCard = new FreelancerCard()
                .withFreelancerId(freelancerId)
                .withAvatarId(cardChanges.getAvatarId() != null ? cardChanges.getAvatarId() : srcCard.getAvatarId())
                .withBriefInfo(cardChanges.getBriefInfo() != null ? cardChanges.getBriefInfo() : srcCard.getBriefInfo())
                .withContacts(cardChanges.getContacts() != null ? cardChanges.getContacts() : srcCard.getContacts())
                .withIsArchived(false)
                .withDeclineReason(emptySet());
        FreelancersCardStatusModerate newStatusModerate = hasModerationChanges(shard, newCard) ? DRAFT : ACCEPTED;
        newCard.withStatusModerate(newStatusModerate);
        FreelancerCardUtils.restoreSiteUrlProtocol(newCard);
        return new FreelancerCardAddOperation(
                singletonList(newCard),
                freelancerCardRepository,
                validationService,
                shard);
    }

    /**
     * Проверяет, есть ли хоть одно значимое для модерации (определяется по полям avatarId, briefInfo и siteUrl) изменение
     * в новой карточке по сравнению с текущей промодерированной карточкой.
     * Если текущей промодерированной карточки ещё не существует, то считается, что значимые для модерации изменения есть.
     */
    private boolean hasModerationChanges(int shard, FreelancerCard newCard) {
        Long freelancerId = newCard.getFreelancerId();
        Map<Long, FreelancerCard> acceptedCardsByFreelancerIds =
                freelancerCardRepository.getAcceptedCardsByFreelancerIds(shard, singletonList(freelancerId));
        FreelancerCard acceptedCard = acceptedCardsByFreelancerIds.get(freelancerId);
        if (acceptedCard == null) {
            return true;
        }
        Long acceptedAvatarId = acceptedCard.getAvatarId();
        String acceptedBriefInfo = acceptedCard.getBriefInfo();
        String acceptedSiteUrl = acceptedCard.getContacts().getSiteUrl();
        Long newAvatarId = newCard.getAvatarId();
        String newBriefInfo = newCard.getBriefInfo();
        String newSiteUrl = newCard.getContacts().getSiteUrl();
        boolean isSensitiveDataEqual = Objects.equals(acceptedAvatarId, newAvatarId)
                && Objects.equals(acceptedBriefInfo, newBriefInfo)
                && Objects.equals(acceptedSiteUrl, newSiteUrl);
        return !isSensitiveDataEqual;
    }

    @SuppressWarnings({"unused", "WeakerAccess"})
    public FreelancerCard getFreelancerCard(ClientId clientId, Long freelancerCardId) {
        int shard = shardHelper.getShardByClientId(clientId);
        List<FreelancerCard> freelancerCards =
                freelancerCardRepository.getFreelancerCards(shard, singletonList(freelancerCardId));
        if (freelancerCards.isEmpty()) {
            return null;
        }
        return freelancerCards.get(0);
    }

    public Result<Long> applyModerationResult(FreelancerCardModeration cardModeration) {
        Long freelancerCardId = cardModeration.getId();
        Long freelancerId = cardModeration.getFreelancerId();
        int shard = shardHelper.getShardByClientId(ClientId.fromLong(freelancerId));
        List<FreelancerCard> freelancerCards =
                freelancerCardRepository.getFreelancerCards(shard, singletonList(freelancerCardId));
        ValidationResult<List<FreelancerCardModeration>, Defect> validationResult =
                validationService.validateCardModerations(singletonList(cardModeration), freelancerCards);
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult);
        }
        checkState(!freelancerCards.isEmpty(), "FreelancerCard (id= %s) isn't found",
                freelancerCardId);
        FreelancerCard freelancerCard = freelancerCards.get(0);
        ModelChanges<FreelancerCard> changes = new ModelChanges<>(freelancerCardId, FreelancerCard.class);
        FreelancersCardStatusModerate moderationStatus = cardModeration.getStatusModerate();
        changes.process(moderationStatus, FreelancerCard.STATUS_MODERATE);
        changes.processNotNull(cardModeration.getDeclineReason(), FreelancerCard.DECLINE_REASON);
        AppliedChanges<FreelancerCard> freelancerCardAppliedChanges = changes.applyTo(freelancerCard);
        freelancerCardRepository.updateFreelancerCards(shard, singletonList(freelancerCardAppliedChanges));
        if (moderationStatus == DECLINED) {
            unarchiveNewestFreelancerCard(freelancerId);
        } else {
            archiveFreelancerCards(freelancerId);
        }
        return Result.successful(cardModeration.getId(), validationResult);
    }

    /**
     * Смотрит на все карточки фрилансера и находит среди них новейшую не архивную со статусом ACCEPTED.
     * Если такой не нашлось, то больше ничего не делает.
     * Если нашлась, то помечает все боле старые (чем найденная) не архивные карточки, как архивные.
     */
    private void archiveFreelancerCards(Long freelancerId) {
        int shard = shardHelper.getShardByClientId(ClientId.fromLong(freelancerId));
        List<FreelancerCard> allActualCards =
                freelancerCardRepository.getNotArchivedFreelancerCards(shard, freelancerId);
        OptionalLong currentFreelancerCardId = StreamEx.of(allActualCards)
                .filter(card -> card.getStatusModerate().equals(ACCEPTED))
                .mapToLong(FreelancerCard::getId)
                .max();
        if (!currentFreelancerCardId.isPresent()) {
            return;
        }
        long maxId = currentFreelancerCardId.getAsLong();
        List<AppliedChanges<FreelancerCard>> appliedChanges = StreamEx.of(allActualCards)
                .filter(card -> card.getId() < maxId)
                .map(card -> {
                    ModelChanges<FreelancerCard> changes = new ModelChanges<>(card.getId(), FreelancerCard.class);
                    changes.process(true, FreelancerCard.IS_ARCHIVED);
                    return changes.applyTo(card);
                })
                .toList();
        if (appliedChanges.isEmpty()) {
            return;
        }
        freelancerCardRepository.updateFreelancerCards(shard, appliedChanges);
    }

    /**
     * Разархивировать последнюю (самую свежую) карточку указанного фрилансера, которая была одобрена модерацией.
     */
    private void unarchiveNewestFreelancerCard(Long freelancerId) {
        int shard = shardHelper.getShardByClientId(ClientId.fromLong(freelancerId));
        Optional<FreelancerCard> acceptedFreelancerCard =
                freelancerCardRepository.getNewestAcceptedFreelancerCard(shard, freelancerId);
        if (!acceptedFreelancerCard.isPresent()) {
            return;
        }
        FreelancerCard freelancerCard = acceptedFreelancerCard.get();
        if (!freelancerCard.getIsArchived()) {
            return;
        }
        ModelChanges<FreelancerCard> changes = new ModelChanges<>(freelancerCard.getId(), FreelancerCard.class);
        changes.process(false, FreelancerCard.IS_ARCHIVED);
        AppliedChanges<FreelancerCard> freelancerCardAppliedChanges = changes.applyTo(freelancerCard);
        freelancerCardRepository.updateFreelancerCards(shard, singletonList(freelancerCardAppliedChanges));
    }

    public List<FreelancerCard> getNewestFreelancerCards(List<Long> freelancerIds) {
        List<FreelancerCard> result = new ArrayList<>(freelancerIds.size());
        shardHelper.groupByShard(freelancerIds, ShardKey.CLIENT_ID)
                .forEach((shard, clientIds) -> result
                        .addAll(freelancerCardRepository.getNewestFreelancerCard(shard, clientIds)));
        return result;
    }


}
