package ru.yandex.direct.core.entity.freelancer.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCertRequest;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificate;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificateType;
import ru.yandex.direct.core.entity.freelancer.service.utils.FreelancerCertificateWithNames;
import ru.yandex.direct.core.entity.freelancer.service.utils.FreelancerCertificatesUtils;
import ru.yandex.direct.core.entity.user.service.BlackboxUserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.expert.client.ExpertClient;
import ru.yandex.direct.expert.client.model.Certificate;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.freelancer.service.utils.FreelancerCertificatesUtils.areNamesEqual;
import static ru.yandex.direct.utils.CollectionUtils.flatToList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Service
@ParametersAreNonnullByDefault
public class FreelancerCertificateService {

    private final ClientService clientService;
    private final ExpertClient expertClient;
    private final BlackboxUserService blackboxUserService;

    @Autowired
    public FreelancerCertificateService(ClientService clientService, ExpertClient expertClient,
                                        BlackboxUserService blackboxUserService) {
        this.clientService = clientService;
        this.expertClient = expertClient;
        this.blackboxUserService = blackboxUserService;
    }

    public Map<Long, List<FreelancerCertificate>> getFlCertsFromExternal(
            Collection<? extends FreelancerCertRequest> requests) {
        Map<ClientId, FreelancerCertRequest> requestByClientId =
                listToMap(requests, fl -> ClientId.fromLong(fl.getFreelancerId()));

        Map<ClientId, Long> uidByClientId = clientService.massGetChiefUidsByClientIds(requestByClientId.keySet());

        List<String> certLogins = StreamEx.of(requests)
                .map(FreelancerCertRequest::getCertLogin)
                .filter(StringUtils::isNotBlank)
                .toList();
        Map<String, Long> uidByLogin = blackboxUserService.getUidsByLogins(certLogins);

        Map<ClientId, List<Long>> uidsByClientId = new HashMap<>(requests.size());
        for (var entry : uidByClientId.entrySet()) {
            List<Long> flUids = new ArrayList<>(2);
            flUids.add(entry.getValue());
            String certLogin = requestByClientId.get(entry.getKey()).getCertLogin();
            if (certLogin != null) {
                Long uidByCertLogin = uidByLogin.get(certLogin);
                if (uidByCertLogin != null) {
                    flUids.add(uidByCertLogin);
                }
            }
            uidsByClientId.put(entry.getKey(), flUids);
        }

        List<Long> allUids = flatToList(uidsByClientId.values());
        Map<Long, List<Certificate>> returnedCerts = expertClient.getCertificates(allUids);
        Map<Long, List<FreelancerCertificateWithNames>> supportedCertificates =
                EntryStream.of(returnedCerts)
                        .mapValues(FreelancerCertificatesUtils::convertSupportedCertificates)
                        .toMap();

        Map<Long, List<FreelancerCertificate>> result = new HashMap<>(requests.size());
        for (var entry : requestByClientId.entrySet()) {
            ClientId clientId = entry.getKey();
            FreelancerCertRequest request = entry.getValue();
            long chiefUid = uidByClientId.get(clientId);
            List<Long> uids = uidsByClientId.getOrDefault(clientId, emptyList());
            ArrayList<FreelancerCertificateWithNames> allNamedCerts = new ArrayList<>();
            for (long uid : uids) {
                List<FreelancerCertificateWithNames> namedCerts = supportedCertificates.getOrDefault(uid, emptyList());
                List<FreelancerCertificateWithNames> validCerts = filterList(namedCerts,
                        c -> uid == chiefUid
                                || areNamesEqual(c, request.getFirstName(), request.getSecondName()));
                allNamedCerts.addAll(validCerts);
            }
            Map<FreelancerCertificateType, List<FreelancerCertificateWithNames>> certsByType =
                    StreamEx.of(allNamedCerts).groupingBy(cert -> cert.getFreelancerCertificate().getType());
            List<FreelancerCertificate> resultCertificates = StreamEx.of(certsByType.values())
                    .filter(CollectionUtils::isNotEmpty)
                    .map(StreamEx::of)
                    //Оставляем только новейшие сертификаты одного типа
                    .map(l -> l.maxBy(FreelancerCertificateWithNames::getConfirmedDate).orElseThrow())
                    .map(FreelancerCertificateWithNames::getFreelancerCertificate)
                    .toList();
            result.put(clientId.asLong(), resultCertificates);
        }

        return result;
    }

}
