package ru.yandex.direct.core.entity.freelancer.service;

import java.io.IOException;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.avatars.client.AvatarsClient;
import ru.yandex.direct.avatars.client.exception.AvatarsClientCommonException;
import ru.yandex.direct.avatars.client.model.AvatarId;
import ru.yandex.direct.core.entity.freelancer.model.ClientAvatar;
import ru.yandex.direct.core.entity.freelancer.model.ClientAvatarId;
import ru.yandex.direct.core.entity.freelancer.model.ClientAvatarsHost;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCard;
import ru.yandex.direct.core.entity.freelancer.repository.ClientAvatarRepository;
import ru.yandex.direct.core.entity.freelancer.repository.FreelancerCardRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.result.ResultState;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.freelancer.service.validation.AvatarsDefects.serverConnectionProblem;
import static ru.yandex.direct.core.entity.freelancer.service.validation.AvatarsDefects.unknownError;
import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.mustHaveFreelancerCard;

/**
 * Сервис сохраняет картинку в Аватарницу МДС, после этого сохраняет идентификатор картинки в карточку фрилансера.
 */
@ParametersAreNonnullByDefault
@Service
public class FreelancerClientAvatarService {
    private static final Logger logger = LoggerFactory.getLogger(ParallelFetcher.class);
    private static final String AVATAR_CONFIG_NAME_SIZE_180 = "size180";
    public static final String DEFAULT_AVATAR_SIZE_180 =
            "https://avatars.mds.yandex.net/get-direct-avatars/1336505/default/" + AVATAR_CONFIG_NAME_SIZE_180;
    private final ShardHelper shardHelper;
    private final FreelancerCardRepository freelancerCardRepository;
    private final FreelancerCardService freelancerCardUpdateService;
    private final ClientAvatarRepository clientAvatarRepository;
    private final AvatarsClientPool freelancersAvatarsClientPool;
    private final AvatarsConfigNameConverter avatarsConfigNameConverter;
    private final ClientAvatarsHost clientAvatarsHost;

    @Autowired
    public FreelancerClientAvatarService(ShardHelper shardHelper,
                                         FreelancerCardRepository freelancerCardRepository,
                                         FreelancerCardService freelancerCardUpdateService,
                                         ClientAvatarRepository clientAvatarRepository,
                                         AvatarsClientPool freelancersAvatarsClientPool,
                                         AvatarsConfigNameConverter avatarsConfigNameConverter) {
        this.shardHelper = shardHelper;
        this.freelancerCardRepository = freelancerCardRepository;
        this.freelancerCardUpdateService = freelancerCardUpdateService;
        this.clientAvatarRepository = clientAvatarRepository;
        this.freelancersAvatarsClientPool = freelancersAvatarsClientPool;
        this.avatarsConfigNameConverter = avatarsConfigNameConverter;
        String avatarsHostConfig = freelancersAvatarsClientPool.getDefaultConfigName();
        clientAvatarsHost = avatarsConfigNameConverter.getHost(avatarsHostConfig);
    }

    /**
     * Возвращает для пар значений ClientId-AvatarId пары значений ClientId-AvatarUrl.
     * Значения извлекаются из таблицы ClientAvatars.
     * Если подходящего значения не найдено, то возвращается аватарка по умолчанию {@link #DEFAULT_AVATAR_SIZE_180},
     * поэтому в выходном наборе всегда столько же пар, сколько в переданном.
     */
    // Размер возвращаемой аватарки специально сделан так хардкорно. Пока ещё не решили где (а соответственно и "как") будут использоваться другие размеры, когда станет ясно надо будет этот метод поменять на более общий.
    public Map<Long, String> massGetUrlSize180(Collection<ClientAvatarId> avatarIds) {
        HashMap<Long, String> result = new HashMap<>(avatarIds.size());
        shardHelper.groupByShard(avatarIds, ShardKey.CLIENT_ID, ClientAvatarId::getClientId)
                .forEach((shard, shardedAvatarIds) -> result.putAll(shardGetUrlSize180(shard, shardedAvatarIds)));
        return result;
    }

    /**
     * Single-версия метода {@link #massGetUrlSize180(Collection)}
     */
    public String getUrlSize180(ClientAvatarId avatarId) {
        return massGetUrlSize180(singleton(avatarId)).get(avatarId.getClientId());
    }

    private Map<Long, String> shardGetUrlSize180(int shard, Collection<ClientAvatarId> avatarIds) {
        List<Long> clientAvatarIds = StreamEx.of(avatarIds)
                .map(ClientAvatarId::getId)
                .filter(Objects::nonNull)
                .toList();
        List<ClientAvatar> clientAvatars = clientAvatarRepository.get(shard, clientAvatarIds);
        Map<Long, String> urlByClientId = StreamEx.of(clientAvatars)
                .mapToEntry(ClientAvatar::getClientId, this::getAvatarUrl)
                .toMap();
        Map<Long, String> result = new HashMap<>(avatarIds.size());
        for (ClientAvatarId clientIdWithAvatarId : avatarIds) {
            Long clientId = clientIdWithAvatarId.getClientId();
            String url = urlByClientId.getOrDefault(clientId, DEFAULT_AVATAR_SIZE_180);
            result.put(clientId, url);
        }
        return result;
    }

    public String getAvatarUrl(ClientAvatar clientAvatar) {
        String configName = avatarsConfigNameConverter.getConfigName(clientAvatar.getHost());
        AvatarsClient avatarsClient = freelancersAvatarsClientPool.getClient(configName);
        String externalId = clientAvatar.getExternalId();
        return avatarsClient.getReadUrl(externalId, AVATAR_CONFIG_NAME_SIZE_180);
    }

    /**
     * Загружает картинку в Аватарницу и сохраняет идентификатор картинки в карточку фрилансера.
     */
    public Result<Long> updateAvatar(ClientId freelancerId, byte[] newAvatar) {
        long longFreelancerId = freelancerId.asLong();
        int shard = shardHelper.getShardByClientId(freelancerId);
        List<FreelancerCard> newestFreelancerCards =
                freelancerCardRepository.getNewestFreelancerCard(shard, singletonList(longFreelancerId));
        if (newestFreelancerCards.isEmpty()) {
            //TODO ajkon: добавить тест на эту ситуацию
            return Result.broken(
                    new ValidationResult<>(longFreelancerId, singletonList(mustHaveFreelancerCard()), emptyList()));
        }
        Result<Long> saveAvatarResult = saveAvatar(freelancerId.asLong(), newAvatar);
        if (!saveAvatarResult.isSuccessful()) {
            return new Result<>(longFreelancerId, saveAvatarResult.getValidationResult(), ResultState.BROKEN);
        }
        Long newAvatarId = saveAvatarResult.getResult();
        FreelancerCard cardChanges = new FreelancerCard()
                .withAvatarId(newAvatarId);
        Result<Long> cardUpdateResult = freelancerCardUpdateService.addChangedFreelancerCard(freelancerId, cardChanges);
        if (!cardUpdateResult.isSuccessful()) {
            return new Result<>(longFreelancerId, cardUpdateResult.getValidationResult(), ResultState.BROKEN);
        }
        return new Result<>(longFreelancerId, cardUpdateResult.getValidationResult(), ResultState.SUCCESSFUL);
    }

    /**
     * Сохраняет картинку без привязки к карточке фрилансера.
     */
    public Result<Long> saveAvatar(Long freelancerId, byte[] newAvatar) {
        int shard = shardHelper.getShardByClientId(ClientId.fromLong(freelancerId));
        AvatarId uploadedAvatarId;
        try {
            uploadedAvatarId = freelancersAvatarsClientPool.getDefaultClient().upload(newAvatar);
        } catch (AvatarsClientCommonException ex) {
            if (ex.getCause() instanceof IOException) {
                ValidationResult<Long, Defect> failedValidationResult =
                        ValidationResult.failed(freelancerId, serverConnectionProblem());
                return Result.broken(failedValidationResult);
            }
            logger.error("Image uploading failed", ex);
            return Result.broken(ValidationResult.failed(freelancerId, unknownError()));
        }
        long clientAvatarId = shardHelper.generateClientAvatarIds(1).get(0);
        ClientAvatar newClientAvatar = new ClientAvatar()
                .withId(clientAvatarId)
                .withClientId(freelancerId)
                .withExternalId(uploadedAvatarId.toAvatarIdString())
                .withHost(clientAvatarsHost)
                .withCreateTime(LocalDateTime.now())
                .withIsDeleted(false);
        List<Long> addResult = clientAvatarRepository.add(shard, singletonList(newClientAvatar));
        checkState(addResult.size() == 1, "ClientAvatarRepository#add didn't return correct newAvatarId.");
        Long newAvatarId = addResult.get(0);
        return Result.successful(newAvatarId);
    }

}
