package ru.yandex.direct.core.entity.freelancer.service;

import java.time.ZonedDateTime;
import java.util.List;
import java.util.UUID;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.freelancer.model.FreelancerFeedback;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerUgcDeclineReason;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerUgcModerationStatus;
import ru.yandex.direct.core.entity.freelancer.service.utils.FreelancerFeedbackConverter;
import ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerFeedbackValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.ugcdb.client.UgcDbClient;
import ru.yandex.direct.ugcdb.client.UgcDbClientException;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.kernel.ugc.protos.direct.TDirectReview;

import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.feedbackNotFound;

@Service
@ParametersAreNonnullByDefault
public class FreelancerFeedbackService {

    private final FreelancerFeedbackValidationService feedbackValidationService;
    private final UgcDbClient ugcDbClient;

    public FreelancerFeedbackService(FreelancerFeedbackValidationService feedbackValidationService,
                                     UgcDbClient ugcDbClient) {
        this.feedbackValidationService = feedbackValidationService;
        this.ugcDbClient = ugcDbClient;
    }

    public Result<String> addFeedbackForFreelancer(ClientId clientId, FreelancerFeedback feedback) {
        feedback.withFeedbackId(generateFeedbackId());

        ValidationResult<FreelancerFeedback, Defect> validationResult =
                feedbackValidationService.validateAddFeedbackForFreelancer(clientId, feedback);
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult);
        }
        String feedbackId = feedback.getFeedbackId();
        try {
            ugcDbClient.saveFeedback(feedback.getFreelancerId(), feedbackId,
                    FreelancerFeedbackConverter.convertFreelancerFeedbackToTDirectReview(feedback));
        } catch (@SuppressWarnings("CaughtExceptionImmediatelyRethrown") UgcDbClientException e) {
            // Ошибка сервера, пробрасываем как есть
            throw e;
        }
        return Result.successful(feedbackId);
    }

    public Result<String> deleteFreelancerFeedback(Long operatorUid, String feedbackId, Long freelancerId) {
        FreelancerFeedback feedback = getFreelancerFeedback(feedbackId, freelancerId);
        if (feedback == null) {
            return Result.broken(ValidationResult.failed(feedbackId, feedbackNotFound()));
        }
        ValidationResult<FreelancerFeedback, Defect> validationResult =
                feedbackValidationService.validateDeleteFreelancerFeedback(operatorUid, feedback);
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult);
        }
        try {
            ugcDbClient.deleteFeedback(freelancerId, feedbackId);
        } catch (@SuppressWarnings("CaughtExceptionImmediatelyRethrown") UgcDbClientException e) {
            // Ошибка сервера, пробрасываем как есть
            throw e;
        }
        return Result.successful(feedbackId);
    }

    public Result<String> updateFreelancerFeedback(Long operatorUid, FreelancerFeedback changedFeedback) {
        String feedbackId = changedFeedback.getFeedbackId();
        FreelancerFeedback feedback = getFreelancerFeedback(feedbackId, changedFeedback.getFreelancerId());
        if (feedback == null) {
            return Result.broken(ValidationResult.failed(feedbackId, feedbackNotFound()));
        }

        FreelancerFeedback updatedFeedback = feedback.withUpdatedTime(ZonedDateTime.now());

        // Проверяем, чтобы не затереть существующие значения
        if (changedFeedback.getFeedbackText() != null) {
            updatedFeedback.withFeedbackText(changedFeedback.getFeedbackText());
        }
        if (changedFeedback.getOverallMark() != null) {
            updatedFeedback.withOverallMark(changedFeedback.getOverallMark());
        }
        if (changedFeedback.getWillRecommend() != null) {
            updatedFeedback.withWillRecommend(changedFeedback.getWillRecommend());
        }

        // Сбрасываем модерацию после обновления отзыва
        updatedFeedback.withModerationStatus(null)
                .withDeclineReasonCode(null);

        ValidationResult<FreelancerFeedback, Defect> validationResult =
                feedbackValidationService.validateUpdateFreelancerFeedback(operatorUid, updatedFeedback);
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult);
        }
        try {
            ugcDbClient.saveFeedback(updatedFeedback.getFreelancerId(), updatedFeedback.getFeedbackId(),
                    FreelancerFeedbackConverter.convertFreelancerFeedbackToTDirectReview(updatedFeedback));
        } catch (@SuppressWarnings("CaughtExceptionImmediatelyRethrown") UgcDbClientException e) {
            // Ошибка сервера, пробрасываем как есть
            throw e;
        }
        return Result.successful(updatedFeedback.getFeedbackId());
    }

    public Result<String> updateFreelancerFeedbackModeration(String feedbackId, Long freelancerId,
                                                             FreelancerUgcModerationStatus status,
                                                             @Nullable FreelancerUgcDeclineReason declineReason) {
        FreelancerFeedback feedback = getFreelancerFeedback(feedbackId, freelancerId);
        if (feedback == null) {
            return Result.broken(ValidationResult.failed(feedbackId, feedbackNotFound()));
        }

        FreelancerFeedback updatedFeedback = feedback.withModerationStatus(status)
                .withDeclineReasonCode(declineReason);
        try {
            ugcDbClient.saveFeedback(updatedFeedback.getFreelancerId(), updatedFeedback.getFeedbackId(),
                    FreelancerFeedbackConverter.convertFreelancerFeedbackToTDirectReview(updatedFeedback));
        } catch (@SuppressWarnings("CaughtExceptionImmediatelyRethrown") UgcDbClientException e) {
            // Ошибка сервера, пробрасываем как есть
            throw e;
        }
        return Result.successful(updatedFeedback.getFeedbackId());
    }

    public FreelancerFeedback getFreelancerFeedback(String feedbackId, Long freelancerId) {
        TDirectReview review;
        try {
            review = ugcDbClient.getFeedback(freelancerId, feedbackId);
        } catch (@SuppressWarnings("CaughtExceptionImmediatelyRethrown") UgcDbClientException e) {
            // Ошибка сервера, пробрасываем как есть
            throw e;
        }
        List<FreelancerFeedback> freelancerFeedbackList =
                FreelancerFeedbackConverter.convertFreelancerFeedbackFromUgcDb(review);
        return freelancerFeedbackList.stream().findFirst().orElse(null);
    }

    public List<FreelancerFeedback> getFreelancerFeedbackList(Long freelancerId) {
        List<TDirectReview> reviewList;
        try {
            reviewList = ugcDbClient.getFeedbackList(freelancerId);
        } catch (@SuppressWarnings("CaughtExceptionImmediatelyRethrown") UgcDbClientException e) {
            // Ошибка сервера, пробрасываем как есть
            throw e;
        }
        return FreelancerFeedbackConverter.convertFreelancerFeedbackFromUgcDb(reviewList);
    }

    private String generateFeedbackId() {
        return UUID.randomUUID().toString();
    }

}
