package ru.yandex.direct.core.entity.freelancer.service;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableSet;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.freelancer.container.FreelancerProjectQueryContainer;
import ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter;
import ru.yandex.direct.core.entity.freelancer.model.Freelancer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerBase;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificate;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerProject;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerProjectIdentity;
import ru.yandex.direct.core.entity.freelancer.repository.FreelancerRepository;
import ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.core.security.AccessDeniedException;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter.allFreelancersIncludingDisabled;
import static ru.yandex.direct.core.entity.freelancer.service.FreelancerService.ACTIVE_PROJECT_STATUSES;
import static ru.yandex.direct.dbutil.model.ClientId.fromLong;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedElement;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.index;

/**
 * Сервис для регистрации фрилансеров.
 * Добавляется запись только во freelancers
 */
@Service
public class FreelancerRegisterService {
    private static final Set<RbacRole> GRANTED_ROLES = ImmutableSet.of(RbacRole.SUPER, RbacRole.MANAGER, RbacRole.SUPPORT);
    /**
     * Максимальное количество представителей (включая главного), которое может быть у клиента,
     * регистрирующегося как фрилансер.
     */
    private static final int MAX_USERS_COUNT = 1;

    private final ShardHelper shardHelper;
    private final RbacService rbacService;
    private final FreelancerRepository freelancerRepository;
    private final FreelancerUpdateService freelancerUpdateService;
    private final FreelancerService freelancerService;
    private final UserService userService;

    public FreelancerRegisterService(ShardHelper shardHelper,
                                     RbacService rbacService,
                                     FreelancerRepository freelancerRepository,
                                     FreelancerUpdateService freelancerUpdateService,
                                     FreelancerService freelancerService,
                                     UserService userService) {
        this.shardHelper = shardHelper;
        this.rbacService = rbacService;
        this.freelancerRepository = freelancerRepository;
        this.freelancerUpdateService = freelancerUpdateService;
        this.freelancerService = freelancerService;
        this.userService = userService;
    }

    /**
     * Регистрация клиента {@code clientId} в качестве фрилансера.
     * {@code operatorUid} используется для проверки роли оператора. Регистрация фрилансеров
     * доступна только Суперам и Менеджерам.
     */
    public Result<Freelancer> registerFreelancer(ClientId clientId, Long operatorUid, Freelancer freelancer) {
        checkOperatorAccess(operatorUid);
        ValidationResult<Freelancer, Defect> validationResult = validate(clientId, freelancer);
        if (validationResult.hasAnyErrors()) {
            return Result.broken(validationResult);
        } else {
            int shard = shardHelper.getShardByClientIdStrictly(clientId);
            freelancerRepository.addFreelancers(shard, Collections.singleton(freelancer));
            return Result.successful(freelancer, validationResult);
        }
    }

    /**
     * Отключение фрилансера. Его данные сохраняются, но он теряет права фрилансера.
     * Разрываем существующие сотрудничества и не даём создавать новые.
     */
    public Result<Long> enableFreelancer(Long freelancerId, Long operatorUid) {
        checkOperatorAccess(operatorUid);

        FreelancersQueryFilter filter = allFreelancersIncludingDisabled().withFreelancerIds(freelancerId).build();
        List<Freelancer> existingFreelancers = freelancerService.getFreelancers(filter);
        if (existingFreelancers.isEmpty()) {
            return Result.broken(ValidationResult.failed(freelancerId, objectNotFound()));
        }
        Freelancer freelancer = existingFreelancers.get(0);

        if (!freelancer.getIsDisabled()) {
            // Фрилансер уже включен, ничего не делаем
            return Result.successful(freelancerId);
        }

        return freelancerUpdateService.updateFreelancer(fromLong(freelancerId),
                new FreelancerBase()
                        .withFreelancerId(freelancerId)
                        .withIsDisabled(false));
    }

    /**
     * Отключение фрилансера. Его данные сохраняются, но он теряет права фрилансера.
     * Разрываем существующие сотрудничества и не даём создавать новые.
     */
    public Result<Long> disableFreelancer(Long freelancerId, Long operatorUid) {
        checkOperatorAccess(operatorUid);

        FreelancersQueryFilter filter = allFreelancersIncludingDisabled().withFreelancerIds(freelancerId).build();
        List<Freelancer> existingFreelancers = freelancerService.getFreelancers(filter);
        if (existingFreelancers.isEmpty()) {
            return Result.broken(ValidationResult.failed(freelancerId, objectNotFound()));
        }
        Freelancer freelancer = existingFreelancers.get(0);

        if (freelancer.getIsDisabled()) {
            // Фрилансер уже отключен, ничего не делаем
            return Result.successful(freelancerId);
        }

        List<FreelancerProject> activeProjects =
                freelancerService.getFreelancersProjects(FreelancerProjectQueryContainer.builder()
                        .withFreelancerIds(freelancerId)
                        .withStatuses(ACTIVE_PROJECT_STATUSES)
                        .build());

        ValidationResult<List<FreelancerProject>, Defect> projectsValidationResult =
                new ValidationResult<>(activeProjects);
        for (int i = 0; i < activeProjects.size(); i++) {
            FreelancerProject activeProject = activeProjects.get(i);
            Result<FreelancerProjectIdentity> result =
                    freelancerService.cancelFreelancerProjectByFreelancer(activeProject);
            if (result.getValidationResult() != null) {
                projectsValidationResult.getOrCreateSubValidationResult(index(i), null)
                        .merge(result.getValidationResult());
            }
        }

        Result<Long> result = freelancerUpdateService.updateFreelancer(fromLong(freelancerId),
                new FreelancerBase()
                        .withFreelancerId(freelancerId)
                        .withIsDisabled(true));

        if (!result.isSuccessful() || projectsValidationResult.hasAnyErrors()) {
            // На самом деле эта валидация не относится к принимаемым аргументам и можно не отдавать её наружу
            ValidationResult<Long, Defect> vr = new ValidationResult<>(freelancerId);
            vr.getOrCreateSubValidationResult(field("projects"), null).merge(projectsValidationResult);
            vr.getOrCreateSubValidationResult(field("freelancer"), null).merge(result.getValidationResult());
            return Result.broken(vr);
        }

        return Result.successful(freelancerId);
    }

    private void checkOperatorAccess(Long operatorUid) {
        RbacRole operatorRole = rbacService.getUidRole(operatorUid);
        if (!GRANTED_ROLES.contains(operatorRole)) {
            throw new AccessDeniedException(
                    String.format("Оператор с ролью %s не допускается к изменению параметров клиента",
                            operatorRole.toString()));
        }
    }

    private ValidationResult<Freelancer, Defect> validate(ClientId clientId, Freelancer freelancer) {
        ModelItemValidationBuilder<Freelancer> vb = ModelItemValidationBuilder.of(freelancer);
        vb.item(Freelancer.FREELANCER_ID)
                .check(notNull())
                .check(CommonConstraints.isEqual(clientId.asLong(), CommonDefects.inconsistentState()))
                .checkBy(getUserContValidator(), When.isValid());
        vb.list(Freelancer.CERTIFICATES)
                .checkEach(unique(FreelancerCertificate::getType), duplicatedElement());
        return vb.getResult();
    }

    private Validator<? extends Long, Defect> getUserContValidator() {
        return freelancerId -> {
            ClientId clientId = fromLong(freelancerId);
            Map<ClientId, List<Long>> usersByClientId =
                    userService.massGetUidsByClientIds(Collections.singletonList(clientId));
            List<Long> users = usersByClientId.get(clientId);
            if (users != null && users.size() > MAX_USERS_COUNT) {
                return ValidationResult.failed(freelancerId, FreelancerDefects.usersCountExceeded());
            }
            return ValidationResult.success(freelancerId);
        };
    }

}
