package ru.yandex.direct.core.entity.freelancer.service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.freelancer.container.FreelancerProjectFilter;
import ru.yandex.direct.core.entity.freelancer.container.FreelancerProjectQueryContainer;
import ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter;
import ru.yandex.direct.core.entity.freelancer.model.Freelancer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerProject;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerProjectIdentity;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerProjectStatus;
import ru.yandex.direct.core.entity.freelancer.repository.FreelancerProjectRepository;
import ru.yandex.direct.core.entity.freelancer.repository.FreelancerRepository;
import ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerValidationService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.rbac.RbacClientsRelations;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.freelancer.container.FreelancerProjectQueryContainer.builder;
import static ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter.enabledFreelancers;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class FreelancerService {
    private final FreelancerProjectRepository freelancerProjectRepository;
    private final FreelancerRepository freelancerRepository;
    private final RbacClientsRelations rbacClientsRelations;
    private final ShardHelper shardHelper;
    private final FreelancerValidationService freelancerValidationService;

    static final Set<FreelancerProjectStatus> ACTIVE_PROJECT_STATUSES =
            ImmutableSet.copyOf(EnumSet.of(FreelancerProjectStatus.NEW, FreelancerProjectStatus.INPROGRESS));
    private static final Set<FreelancerProjectStatus> INACTIVE_PROJECT_STATUSES;

    static {
        EnumSet<FreelancerProjectStatus> inactiveStatues = EnumSet.allOf(FreelancerProjectStatus.class);
        inactiveStatues.removeAll(ACTIVE_PROJECT_STATUSES);
        INACTIVE_PROJECT_STATUSES = ImmutableSet.copyOf(inactiveStatues);
    }

    @Autowired
    public FreelancerService(FreelancerProjectRepository freelancerProjectRepository,
                             FreelancerRepository freelancerRepository,
                             ShardHelper shardHelper,
                             FreelancerValidationService freelancerValidationService,
                             RbacClientsRelations rbacClientsRelations) {
        this.freelancerProjectRepository = freelancerProjectRepository;
        this.freelancerRepository = freelancerRepository;
        this.shardHelper = shardHelper;
        this.freelancerValidationService = freelancerValidationService;
        this.rbacClientsRelations = rbacClientsRelations;
    }

    /**
     * Получение фрилансеров с карточками для списка фрилансеров
     */
    public List<Freelancer> getFreelancers(FreelancersQueryFilter filter) {
        return getFreelancers(filter, LimitOffset.maxLimited());
    }

    /**
     * Получение фрилансеров с карточками для списка фрилансеров
     */
    public List<Freelancer> getFreelancers(FreelancersQueryFilter filter, LimitOffset limitOffset) {
        List<Freelancer> result = new ArrayList<>();
        shardHelper.forEachShard(shard -> result.addAll(getFreelancersFromShardStrictly(shard, filter)));
        return result.stream()
                .skip(limitOffset.offset())
                .limit(limitOffset.limit())
                .collect(toList());
    }

    /**
     * Получить фрилансеров из шарда {@code shard} с учётом метабазы
     */
    private List<Freelancer> getFreelancersFromShardStrictly(int shard, FreelancersQueryFilter filter) {
        List<Freelancer> freelancers = freelancerRepository.getByFilter(shard, filter);
        // в тестинге из-за телепортации копия фрилансера может оказаться в соседнем шарде
        // оставляем только тех, кто зарегистрирован в текущем шарде
        var shardsByClientIds = shardHelper.getShardsByClientIds(mapList(freelancers, Freelancer::getId));
        Set<Long> validFreelancerIds = EntryStream.of(shardsByClientIds)
                .filterValues(s -> s == shard)
                .keys().toSet();
        return freelancers.stream()
                .filter(it -> validFreelancerIds.contains(it.getId()))
                .collect(toList());
    }

    /**
     * Получение фрилансеров по их {@code clientId}
     */
    public List<Freelancer> getFreelancers(Collection<Long> ids) {
        return shardHelper.groupByShard(ids, ShardKey.CLIENT_ID)
            .stream()
            .mapKeyValue(freelancerRepository::getByIds)
            .flatMap(Collection::stream)
            .toList();
    }

    /**
     * Проверка, является ли клиент с {@code clientId} специалистом-фрилансером
     */
    public boolean isFreelancer(ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        FreelancersQueryFilter filter = enabledFreelancers().withFreelancerIds(clientId.asLong()).build();
        List<Freelancer> foundFreelancers = freelancerRepository.getByFilter(shard, filter);
        return !foundFreelancers.isEmpty();
    }

    /**
     * Получение фрилансера с последней версией карточки, независимо от её статуса модерации
     */
    public Freelancer getFreelancerWithNewestCard(Long freelancerId) {
        int shard = shardHelper.getShardByClientId(ClientId.fromLong(freelancerId));
        return freelancerRepository.getFreelancerWithNewestCard(shard, freelancerId);
    }

    /**
     * Получение проектов для списка фрилансеров
     * Ищем по всем шардам, т.к. проекты хранятся в шардах клиентов
     */
    public List<FreelancerProject> getFreelancersProjects(List<Long> freelancerIds) {
        if (freelancerIds.isEmpty()) {
            return emptyList();
        }
        FreelancerProjectQueryContainer queryContainer = builder().withFreelancerIds(freelancerIds).build();
        return getFreelancersProjects(queryContainer);
    }

    /**
     * Получение проектов по заданным критериям
     */
    List<FreelancerProject> getFreelancersProjects(FreelancerProjectQueryContainer queryContainer) {
        List<FreelancerProject> result = new ArrayList<>();
        shardHelper.forEachShard(shard -> {
            List<FreelancerProject> projects =
                    freelancerProjectRepository.get(shard, queryContainer, LimitOffset.maxLimited());
            result.addAll(projects);
        });
        return result;
    }

    public List<FreelancerProject> getClientProjects(Long clientId, FreelancerProjectFilter filter) {
        int shard = shardHelper.getShardByClientId(ClientId.fromLong(clientId));
        FreelancerProjectQueryContainer queryContainer = getFreelancerProjectQueryContainer(List.of(clientId), filter);
        LimitOffset limitOffset =
                filter.getLimit() != null ? LimitOffset.limited(filter.getLimit()) : LimitOffset.maxLimited();
        return freelancerProjectRepository.get(shard, queryContainer, limitOffset);
    }

    public List<FreelancerProject> massGetClientProjects(List<Long> clientIds, FreelancerProjectFilter filter) {
        FreelancerProjectQueryContainer queryContainer = getFreelancerProjectQueryContainer(clientIds, filter);
        LimitOffset limitOffset =
                filter.getLimit() != null ? LimitOffset.limited(filter.getLimit()) : LimitOffset.maxLimited();
        List<FreelancerProject> result = new ArrayList<>();
        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID).forEach(
                (shard, clientIdsGroupByShard) -> result.addAll(
                        freelancerProjectRepository.get(shard, queryContainer, limitOffset)));
        return result;
    }

    private FreelancerProjectQueryContainer getFreelancerProjectQueryContainer(List<Long> clientIds,
                                                                               FreelancerProjectFilter filter) {
        FreelancerProjectQueryContainer.Builder queryBuilder = FreelancerProjectQueryContainer.builder();
        queryBuilder.withClientIds(clientIds);
        if (filter.getFreelancerId() != null) {
            queryBuilder.withFreelancerIds(singletonList(filter.getFreelancerId()));
        }
        if (filter.getIsActive() != null) {
            queryBuilder.withStatuses(filter.getIsActive() ? ACTIVE_PROJECT_STATUSES : INACTIVE_PROJECT_STATUSES);
        }
        if (filter.getIsStarted() != null) {
            queryBuilder.withStarted(filter.getIsStarted());
        }
        return queryBuilder.build();
    }

    public List<FreelancerProject> getFreelancersProjectsByProjectIds(List<Long> projectIds) {
        if (projectIds.isEmpty()) {
            return emptyList();
        }
        List<FreelancerProject> result = new ArrayList<>();
        shardHelper.forEachShard(shard -> {
            FreelancerProjectQueryContainer queryContainer =
                    builder().withProjectIds(Iterables.toArray(projectIds, Long.class)).build();
            List<FreelancerProject> projects =
                    freelancerProjectRepository.get(shard, queryContainer, LimitOffset.maxLimited());
            result.addAll(projects);
        });
        return result;
    }

    /**
     * Создать запрос на работу с фрилансером. Под капотом создаётся прооект. По бизнес логике этот метод позволяет создать заявку клиентом clientId на управление его кампаниями фрилансером freelancerId </br>
     * Проект будет создан в шарде клиента со статусом {@link FreelancerProjectStatus#NEW}
     *
     * @param client       клиент, который запрашивает доступ для фрилансера
     * @param freelancerId фрилансер, которому уйдёт заявка
     * @return id созданного проекта
     */
    public Result<Long> requestFreelancerService(User client, Long freelancerId) {
        ValidationResult<Void, Defect> vr =
                freelancerValidationService.validateProjectRequest(client, freelancerId);
        if (vr.hasAnyErrors()) {
            return Result.broken(vr);
        }
        int shard = shardHelper.getShardByClientId(client.getClientId());

        long newProjectId = shardHelper.generateFreelancerProjectIds(1).get(0);
        LocalDateTime now = LocalDateTime.now();
        FreelancerProject project = new FreelancerProject()
                .withId(newProjectId)
                .withClientId(client.getClientId().asLong())
                .withFreelancerId(freelancerId)
                .withCreatedTime(now)
                .withUpdatedTime(now)
                .withStatus(FreelancerProjectStatus.NEW);
        freelancerProjectRepository.add(shard, singletonList(project));
        return Result.successful(newProjectId);
    }

    /**
     * Принять запрос клиента на услуги фрилансера. Добавляется связка в client_relations
     *
     * @see RbacClientsRelations#addFreelancerRelation(ClientId, ClientId)
     */
    public Result<FreelancerProjectIdentity> acceptFreelancerProject(FreelancerProjectIdentity projectIdentity) {
        ValidationResult<FreelancerProjectIdentity, Defect> identityVr =
                freelancerValidationService.validateProjectIdentity(projectIdentity);
        if (identityVr.hasAnyErrors()) {
            return Result.broken(identityVr);
        }
        FreelancerProjectStatus newStatus = FreelancerProjectStatus.INPROGRESS;
        Map<Long, FreelancerProject> existingProjects = getFreelancerProjects(projectIdentity);
        ValidationResult<Long, Defect> statusVr =
                freelancerValidationService
                        .validateChangeProjectStatusByFreelancer(projectIdentity, newStatus, existingProjects);
        if (statusVr.hasAnyErrors()) {
            return Result.broken(statusVr);
        }

        rbacClientsRelations.addFreelancerRelation(ClientId.fromLong(projectIdentity.getClientId()),
                ClientId.fromLong(projectIdentity.getFreelancerId()));
        LocalDateTime now = LocalDateTime.now();
        updateFreelancerProjectStatus(existingProjects.get(projectIdentity.getId()), newStatus, now, now);
        return Result.successful(projectIdentity);
    }

    /**
     * Отклонить взаимоотношения фрилансера с клиентом со стороны фрилансера. Будет удалена связка из client_relations
     *
     * @see RbacClientsRelations#removeFreelancerRelation(ClientId, ClientId)
     */
    public Result<FreelancerProjectIdentity> cancelFreelancerProjectByFreelancer(
            FreelancerProjectIdentity projectIdentity) {
        ValidationResult<FreelancerProjectIdentity, Defect> identityVr =
                freelancerValidationService.validateProjectIdentity(projectIdentity);
        if (identityVr.hasAnyErrors()) {
            return Result.broken(identityVr);
        }
        FreelancerProjectStatus newStatus = FreelancerProjectStatus.CANCELLEDBYFREELANCER;
        Map<Long, FreelancerProject> existingProjects = getFreelancerProjects(projectIdentity);
        ValidationResult<Long, Defect> statusVr =
                freelancerValidationService
                        .validateChangeProjectStatusByFreelancer(projectIdentity, newStatus, existingProjects);
        if (statusVr.hasAnyErrors()) {
            return Result.broken(statusVr);
        }

        rbacClientsRelations.removeFreelancerRelation(ClientId.fromLong(projectIdentity.getClientId()),
                ClientId.fromLong((projectIdentity.getFreelancerId())));
        updateFreelancerProjectStatus(existingProjects.get(projectIdentity.getId()), newStatus, LocalDateTime.now(),
                null);
        return Result.successful(projectIdentity);
    }

    /**
     * Отклонить взаимоотношения фрилансера с клиентом со стороны клиента. Будет удалена связка из client_relations
     *
     * @see RbacClientsRelations#removeFreelancerRelation(ClientId, ClientId)
     */
    public Result<FreelancerProjectIdentity> cancelFreelancerProjectByClient(FreelancerProjectIdentity projectIdentity) {
        ValidationResult<FreelancerProjectIdentity, Defect> identityVr =
                freelancerValidationService.validateProjectIdentity(projectIdentity);
        if (identityVr.hasAnyErrors()) {
            return Result.broken(identityVr);
        }
        FreelancerProjectStatus newStatus = FreelancerProjectStatus.CANCELLEDBYCLIENT;
        Map<Long, FreelancerProject> existingProjects = getFreelancerProjects(projectIdentity);
        ValidationResult<Long, Defect> statusVr =
                freelancerValidationService
                        .validateChangeProjectStatusByClient(projectIdentity, newStatus, existingProjects);
        if (statusVr.hasAnyErrors()) {
            return Result.broken(statusVr);
        }

        rbacClientsRelations.removeFreelancerRelation(ClientId.fromLong(projectIdentity.getClientId()),
                ClientId.fromLong((projectIdentity.getFreelancerId())));
        updateFreelancerProjectStatus(existingProjects.get(projectIdentity.getId()), newStatus, LocalDateTime.now(),
                null);
        return Result.successful(projectIdentity);
    }

    public boolean isAnyFreelancerRelation(ClientId clientId, ClientId freelancerId) {
        return rbacClientsRelations.getFreelancerRelation(clientId, freelancerId).getRelationType() != null;
    }

    /**
     * Получение проекта фрилансером
     * Смотрим в шарде клиента
     */
    private Map<Long, FreelancerProject> getFreelancerProjects(FreelancerProjectIdentity projectIdentity) {
        int shard = shardHelper.getShardByClientId(ClientId.fromLong(projectIdentity.getClientId()));
        FreelancerProjectQueryContainer queryContainer = builder()
                .withClientIds(projectIdentity.getClientId())
                .withFreelancerIds(projectIdentity.getFreelancerId())
                .withProjectIds(projectIdentity.getId())
                .build();
        List<FreelancerProject> freelancerProjects =
                freelancerProjectRepository.get(shard, queryContainer, LimitOffset.maxLimited());
        return listToMap(freelancerProjects, FreelancerProject::getId);

    }

    /**
     * Апдейтим проект в шарде клиента
     */
    private void updateFreelancerProjectStatus(FreelancerProject project,
                                               FreelancerProjectStatus status,
                                               LocalDateTime updateTime,
                                               @Nullable LocalDateTime startedTime) {
        int shard = shardHelper.getShardByClientId(ClientId.fromLong(project.getClientId()));
        ModelChanges<FreelancerProject> changes = new ModelChanges<>(project.getId(), FreelancerProject.class);
        changes.process(status, FreelancerProject.STATUS);
        changes.process(updateTime, FreelancerProject.UPDATED_TIME);
        changes.processNotNull(startedTime, FreelancerProject.STARTED_TIME);
        freelancerProjectRepository.update(shard, singletonList(changes.applyTo(project)));
    }
}
