package ru.yandex.direct.core.entity.freelancer.service;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.freelancer.model.FreelancerSkill;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerSkillOffer;
import ru.yandex.direct.core.entity.freelancer.operation.FreelancerSkillOffersSetOperation;
import ru.yandex.direct.core.entity.freelancer.repository.FreelancerSkillsRepository;
import ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerSkillOffersValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.result.MassResult;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.core.entity.freelancer.model.FreelancerSkill.CAMPAIGN_CONDUCTING;
import static ru.yandex.direct.core.entity.freelancer.model.FreelancerSkill.SETTING_UP_CAMPAIGNS_FROM_SCRATCH;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Service
@ParametersAreNonnullByDefault
public class FreelancerSkillOffersService {
    private static final Comparator<FreelancerSkillOffer> FREELANCER_SKILL_OFFER_COMPARATOR =
            createSkillOfferComparator();

    private final FreelancerSkillsRepository freelancerSkillsRepository;
    private final ShardHelper shardHelper;
    private final FreelancerSkillOffersValidationService freelancerSkillOffersValidationService;

    @Autowired
    public FreelancerSkillOffersService(
            ShardHelper shardHelper,
            FreelancerSkillsRepository freelancerSkillsRepository,
            FreelancerSkillOffersValidationService freelancerSkillOffersValidationService) {
        this.shardHelper = shardHelper;
        this.freelancerSkillsRepository = freelancerSkillsRepository;
        this.freelancerSkillOffersValidationService = freelancerSkillOffersValidationService;
    }

    private static Comparator<FreelancerSkillOffer> createSkillOfferComparator() {
        //Идентификаторы приоритетных услуг при выборе главной услуги фрилансера.
        //Перечень приоритетных услуг взят отсюда: https://st.yandex-team.ru/DIRECT-86372#1540481548000
        Set<Long> prioritySkillIds =
                StreamEx.of(SETTING_UP_CAMPAIGNS_FROM_SCRATCH, CAMPAIGN_CONDUCTING)
                        .map(FreelancerSkill::getSkillId)
                        .toSet();
        // Сравнивает услуги фриланскеров по принадлежности к приоритетным. Приоритетные услуги имеют меньший вес при
        // сравнении.
        Comparator<FreelancerSkillOffer> prioritySkillOfferComparator =
                Comparator.comparing(offer ->
                        !prioritySkillIds.contains(offer.getSkillId()));
        return prioritySkillOfferComparator
                .thenComparing(FreelancerSkillOffer::getPrice)
                .thenComparing(offer -> FreelancerSkill.getById(offer.getSkillId()).getDescription());
    }

    public List<FreelancerSkillOffer> getFreelancerSkillsOffers(List<Long> freelancerIds) {
        List<FreelancerSkillOffer> result = new ArrayList<>(freelancerIds.size());
        shardHelper.groupByShard(freelancerIds, ShardKey.CLIENT_ID)
                .forEach((shard, ids) -> result.addAll(freelancerSkillsRepository.getOffers(shard, ids)));

        Set<Long> activeSkillIds = listToSet(FreelancerSkill.allFreelancerSkills(), FreelancerSkill::getSkillId);
        return StreamEx.of(result)
                .filter(offer -> activeSkillIds.contains(offer.getSkillId()))
                .toList();
    }

    /**
     * Возвращает главную услугу фрилансера выбирая её по следующим приоритетм (в порядке убвания важности):
     * 1. Приоритетная услуга.
     * 2. Наименьшая цена.
     * 3. Первая по алфавиту.
     * Перечень приоритетных услуг взят отсюда:
     * <a href="https://st.yandex-team.ru/DIRECT-86372#1540481548000">https://st.yandex-team.ru/DIRECT-86372#1540481548000</a>
     */
    @Nullable
    public FreelancerSkillOffer getFreelancerMainSkillsOffer(Long freelancerId) {
        int shard = shardHelper.getShardByClientId(ClientId.fromLong(freelancerId));
        List<FreelancerSkillOffer> offers =
                freelancerSkillsRepository.getOffers(shard, Collections.singletonList(freelancerId));
        if (offers.isEmpty()) {
            return null;
        }
        return Collections.min(offers, FREELANCER_SKILL_OFFER_COMPARATOR);
    }

    /**
     * Добавляет и/или обновляет услугу в списке услуг выполняемых фрилансером.
     *
     * @param freelancerId      Пользователь для которого делаем изменения.
     * @param clientSkillOffers Услуги фрилансера которые надо добавить или обновить. Поля freelancerId должны
     *                          совпадать с переданным параметром clientId.
     * @return результат обновлённия или добавленния услуги
     */
    public MassResult<FreelancerSkillOffer> setFreelancerSkillOffer(ClientId freelancerId,
                                                                    List<FreelancerSkillOffer> clientSkillOffers) {
        Long longClientId = freelancerId.asLong();
        for (FreelancerSkillOffer offer : clientSkillOffers) {
            //страхуемся от неправильного формирования параметров этого метода
            checkArgument(longClientId.equals(offer.getFreelancerId()),
                    "skillOffers#getFreelancerId()=%1$d is not equal ClientId=%2$d", offer.getFreelancerId(),
                    longClientId);
        }
        int shard = shardHelper.getShardByClientId(ClientId.fromLong(longClientId));
        FreelancerSkillOffersSetOperation skillSetOperation =
                new FreelancerSkillOffersSetOperation(shard, clientSkillOffers, freelancerSkillOffersValidationService,
                        freelancerSkillsRepository);
        return skillSetOperation.prepareAndApply();
    }

    public int deleteFreelancerSkillsOffer(ClientId clientId, List<Long> skillIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        return freelancerSkillsRepository.delete(shard, clientId, skillIds);
    }
}
