package ru.yandex.direct.core.entity.freelancer.service;

import java.util.Collections;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.freelancer.model.FreelancerBase;
import ru.yandex.direct.core.entity.freelancer.operation.FreelancerUpdateOperation;
import ru.yandex.direct.core.entity.freelancer.repository.FreelancerRepository;
import ru.yandex.direct.core.entity.freelancer.service.validation.UpdateFreelancerValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.result.Result;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * Сервис для модификации фрилансеров.
 * <p>
 * Отделён от {@link FreelancerService}, так как зависит от validation service'ов, которые зависят от {@link FreelancerService}
 */
@Service
public class FreelancerUpdateService {
    private final ShardHelper shardHelper;
    private final FreelancerRepository freelancerRepository;
    private final UpdateFreelancerValidationService updateFreelancerValidationService;

    public FreelancerUpdateService(ShardHelper shardHelper,
                                   FreelancerRepository freelancerRepository,
                                   UpdateFreelancerValidationService updateFreelancerValidationService) {
        this.shardHelper = shardHelper;
        this.freelancerRepository = freelancerRepository;
        this.updateFreelancerValidationService = updateFreelancerValidationService;
    }

    public Result<Long> updateFreelancer(ClientId clientId, FreelancerBase freelancerBase) {
        FreelancerUpdateOperation updateOperation = getFreelancerUpdateOperation(clientId, freelancerBase);

        return updateOperation.prepareAndApply().get(0);
    }

    public FreelancerUpdateOperation getFreelancerUpdateOperation(ClientId clientId, FreelancerBase freelancer) {
        checkArgument(freelancer.getId() != null, "ID must be specified");
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        ModelChanges<FreelancerBase> modelChanges = new ModelChanges<>(freelancer.getId(), FreelancerBase.class);
        modelChanges.processNotNull(freelancer.getIsSearchable(), FreelancerBase.IS_SEARCHABLE);
        modelChanges.processNotNull(freelancer.getIsDisabled(), FreelancerBase.IS_DISABLED);
        modelChanges.processNotNull(freelancer.getCertificates(), FreelancerBase.CERTIFICATES);
        modelChanges.processNotNull(freelancer.getRating(), FreelancerBase.RATING);
        modelChanges.processNotNull(freelancer.getFirstName(), FreelancerBase.FIRST_NAME);
        modelChanges.processNotNull(freelancer.getSecondName(), FreelancerBase.SECOND_NAME);
        modelChanges.processNotNull(freelancer.getCertLogin(), FreelancerBase.CERT_LOGIN);
        modelChanges.processNotNull(freelancer.getRegionId(), FreelancerBase.REGION_ID);
        modelChanges.processNotNull(freelancer.getFeedbackCount(), FreelancerBase.FEEDBACK_COUNT);
        modelChanges.processNotNull(freelancer.getStatus(), FreelancerBase.STATUS);
        modelChanges.processNotNull(freelancer.getAdvQualityRank(), FreelancerBase.ADV_QUALITY_RANK);
        modelChanges.processNotNull(freelancer.getAdvQualityRating(), FreelancerBase.ADV_QUALITY_RATING);

        return new FreelancerUpdateOperation(Collections.singletonList(modelChanges),
                freelancerRepository,
                updateFreelancerValidationService,
                shard);
    }

}
