package ru.yandex.direct.core.entity.freelancer.service.utils;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.UnaryOperator;

import com.google.common.collect.ImmutableSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificate;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificateType;
import ru.yandex.direct.expert.client.model.Certificate;

import static java.util.Collections.emptyList;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificateType.DIRECT;
import static ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificateType.DIRECT_PRO;
import static ru.yandex.direct.core.entity.freelancer.model.FreelancerCertificateType.METRIKA;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.notEquals;

public class FreelancerCertificatesUtils {
    private static final Set<String> SUPPORTED_CERTIFICATE_TYPES =
            ImmutableSet.of(DIRECT_PRO.name(), DIRECT.name(), METRIKA.name());

    private static final UnaryOperator<String> NORMALIZE_NAME = str -> str.replace('ё', 'е').replace('Ё', 'Е');

    private static final Logger logger = LoggerFactory.getLogger(FreelancerCertificatesUtils.class);
    // С 24 июля 2019-го за простой тест без прокторинга сертификат больше не дают:
    // https://yandex.ru/adv/news/testirovanie-po-direktu-s-proktoringom-vykhodit-iz-bety-i-stanovitsya-osnovnym
    private static final LocalDateTime DIRECT_TYPE_LAST_DATE = LocalDateTime.of(2019, 7, 24, 0, 0);
    private static final String DIRECT_TYPE = DIRECT.name();

    /**
     * Оставляет только поддерживаемые у специалистов сертификаты и конвертирует их в тип FreelancerCertificate.
     */
    public static List<FreelancerCertificateWithNames> convertSupportedCertificates(
            List<Certificate> certificates) {
        if (isEmpty(certificates)) {
            return emptyList();
        }
        ArrayList<FreelancerCertificateWithNames> freelancerCertificates = new ArrayList<>();
        for (Certificate certificate : certificates) {
            String certType = certificate.getExam().getSlug().replace("-", "_").toUpperCase();
            LocalDateTime confirmedDate = certificate.getConfirmedDate();
            if (SUPPORTED_CERTIFICATE_TYPES.contains(certType) && certificate.getActive().equals(1)
                    && (notEquals(certType, DIRECT_TYPE) || confirmedDate.isBefore(DIRECT_TYPE_LAST_DATE))) {
                FreelancerCertificate cert = new FreelancerCertificate()
                        .withCertId(Long.valueOf(certificate.getCertId()))
                        .withType(FreelancerCertificateType.valueOf(certType));
                freelancerCertificates.add(new FreelancerCertificateWithNames()
                        .withFreelancerCertificate(cert)
                        .withFirstName(certificate.getFirstname())
                        .withSecondName(certificate.getLastname())
                        .withConfirmedDate(confirmedDate)
                );
                checkCertStatus(certificate);
            }
        }
        return freelancerCertificates;
    }

    public static boolean isDirectType(FreelancerCertificateWithNames certWithNames) {
        FreelancerCertificate cert = certWithNames.getFreelancerCertificate();
        return cert.getType() == DIRECT || cert.getType() == DIRECT_PRO;
    }

    // TODO maxlog@ : летом 2019-го проверить логи, не появилось ли варнингов про несоответствие статуса и дат из
    //  этого класса
    private static void checkCertStatus(Certificate certificate) {
        if (certificate == null) {
            return;
        }
        LocalDateTime dueDate = certificate.getDueDate();
        LocalDateTime confirmedDate = certificate.getConfirmedDate();
        if (dueDate == null || confirmedDate == null) {
            return;
        }
        LocalDateTime now = LocalDateTime.now();
        boolean isValidByDates = confirmedDate.isBefore(now) && dueDate.isAfter(now);
        boolean isActive = certificate.getActive().equals(1);
        if (isValidByDates != isActive) {
            logger.warn(
                    "Active status dose not correspond the dates. active={}; confirmedDate={}; dueDate = {}; now={}",
                    certificate.getActive(),
                    confirmedDate,
                    dueDate,
                    now);
        }
    }

    public static boolean areNamesEqual(FreelancerCertificateWithNames cert, String firstName, String secondName) {
        return areNamesNormalizedEqual(firstName, cert.getFirstName())
                && areNamesNormalizedEqual(secondName, cert.getSecondName());
    }

    private static boolean areNamesNormalizedEqual(String name1, String name2) {
        String normalizedName = ifNotNull(name1, NORMALIZE_NAME);
        String normalizedName2 = ifNotNull(name2, NORMALIZE_NAME);
        return Objects.equals(normalizedName, normalizedName2);
    }
}
