package ru.yandex.direct.core.entity.freelancer.service.utils;

import java.time.Instant;
import java.time.ZonedDateTime;
import java.util.Collections;
import java.util.List;

import javax.annotation.Nonnull;

import NUgc.NSchema.Moderation;
import com.google.protobuf.ProtocolStringList;
import com.google.protobuf.Timestamp;

import ru.yandex.direct.core.entity.freelancer.model.FreelancerFeedback;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerSkill;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerUgcDeclineReason;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerUgcModerationStatus;
import ru.yandex.kernel.ugc.protos.direct.TDirectReview;
import ru.yandex.misc.lang.StringUtils;

import static NUgc.NSchema.Moderation.TModerationStatus;
import static NUgc.NSchema.Moderation.TReviewModerationData;
import static NUgc.NSchema.Moderation.TTextDeclineReason;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.DateTimeUtils.MSK;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * @implNote В UGC DB время хранится в UTC, во внутренней модели храним по московской таймзоне
 */
public class FreelancerFeedbackConverter {

    public static List<FreelancerFeedback> convertFreelancerFeedbackFromUgcDb(TDirectReview ugcDbResponse) {
        if (ugcDbResponse == null) {
            return emptyList();
        }
        return Collections.singletonList(convertFreelancerFeedbackFromTDirectReview(ugcDbResponse));
    }

    public static List<FreelancerFeedback> convertFreelancerFeedbackFromUgcDb(List<TDirectReview> reviewsList) {
        if (reviewsList == null) {
            return emptyList();
        }
        return mapList(reviewsList, FreelancerFeedbackConverter::convertFreelancerFeedbackFromTDirectReview);
    }

    private static FreelancerFeedback convertFreelancerFeedbackFromTDirectReview(TDirectReview directReview) {
        if (directReview == null) {
            return null;
        }
        FreelancerFeedback feedback = new FreelancerFeedback()
                .withFeedbackId(directReview.getReviewId())
                .withAuthorUid(directReview.getAuthorUserId())
                .withFreelancerId(Long.valueOf(directReview.getContractorId()))
                .withCreatedTime(convertZonedDateTimeFromTimestamp(directReview.getCreateTime()))
                .withUpdatedTime(convertZonedDateTimeFromTimestamp(directReview.getUpdatedTime()))
                .withOverallMark((long) directReview.getOverallMark())
                .withFeedbackText(directReview.getFeedbackText())
                .withSkills(convertFreelancerSkillsFromUgc(directReview.getSkillIdsList()))
                .withWillRecommend(convertWillRecommendFromUgc(directReview.getRecommendation()));
        if (directReview.hasModeration()) {
            feedback.withModerationStatus(convertModerationStatusFromUgc(directReview.getModeration().getStatus()))
                    .withDeclineReasonCode(
                            convertDeclineReasonFromUgc(directReview.getModeration().getDeclineReason()));
        }
        return feedback;
    }

    public static TDirectReview convertFreelancerFeedbackToTDirectReview(FreelancerFeedback feedback) {

        TDirectReview.Builder builder = TDirectReview.newBuilder()
                .setReviewId(feedback.getFeedbackId())
                .setAuthorUserId(feedback.getAuthorUid().intValue())
                .setContractorId(feedback.getFreelancerId().toString())
                .setCreateTime(convertZonedDateTimeToTimestamp(feedback.getCreatedTime()))
                .setUpdatedTime(convertZonedDateTimeToTimestamp(feedback.getUpdatedTime()))
                .setOverallMark(feedback.getOverallMark().intValue())
                .setFeedbackText(feedback.getFeedbackText())
                .addAllSkillIds(mapList(feedback.getSkills(), skill -> skill.getSkillId().toString()))
                .setRecommendation(convertWillRecommendToUgc(feedback.getWillRecommend()));
        if (feedback.getModerationStatus() != null) {
            builder.setModeration(
                    convertModerationDataToUgc(feedback.getModerationStatus(), feedback.getDeclineReasonCode()));
        }
        return builder.build();
    }

    private static FreelancerUgcModerationStatus convertModerationStatusFromUgc(
            @Nonnull Moderation.TModerationStatus.EType ugcModerationStatus) {
        switch (ugcModerationStatus) {
            case IN_PROGRESS:
                return FreelancerUgcModerationStatus.IN_PROGRESS;
            case ACCEPTED:
                return FreelancerUgcModerationStatus.ACCEPTED;
            case DECLINED:
                return FreelancerUgcModerationStatus.DECLINED;
            default:
                throw new IllegalArgumentException("Unexpected enum value " + ugcModerationStatus);
        }
    }

    private static FreelancerUgcDeclineReason convertDeclineReasonFromUgc(
            Moderation.TTextDeclineReason.EType ugcDeclineReason) {
        switch (ugcDeclineReason) {
            case NONE:
                return null;
            case OBSCENE:
                return FreelancerUgcDeclineReason.OBSCENE;
            case PERSONAL_DATA:
                return FreelancerUgcDeclineReason.PERSONAL_DATA;
            case TOO_SHORT:
                return FreelancerUgcDeclineReason.TOO_SHORT;
            case WRONG_LANG:
                return FreelancerUgcDeclineReason.WRONG_LANG;
            case NOT_OPINION:
                return FreelancerUgcDeclineReason.NOT_OPINION;
            case UPPERCASE:
                return FreelancerUgcDeclineReason.UPPERCASE;
            default:
                throw new IllegalStateException("Unexpected enum value " + ugcDeclineReason);
        }
    }

    private static TReviewModerationData convertModerationDataToUgc(@Nonnull FreelancerUgcModerationStatus status,
                                                                    FreelancerUgcDeclineReason declineReason) {
        TModerationStatus.EType tModerationStatus = TModerationStatus.EType.valueOf(status.name());
        TReviewModerationData.Builder builder = TReviewModerationData.newBuilder().setStatus(tModerationStatus);
        if (declineReason != null && status.equals(FreelancerUgcModerationStatus.DECLINED)) {
            TTextDeclineReason.EType tTextDeclineReason = TTextDeclineReason.EType.valueOf(declineReason.name());
            builder.setDeclineReason(tTextDeclineReason);
        } else {
            builder.setDeclineReason(TTextDeclineReason.EType.NONE);
        }
        return builder.build();
    }

    private static List<FreelancerSkill> convertFreelancerSkillsFromUgc(ProtocolStringList skillIdProtocolStringList) {
        List<Long> skillIdList = mapList(skillIdProtocolStringList, Long::valueOf);
        return mapList(skillIdList, FreelancerFeedbackConverter::convertLongToFreelancerSkill);
    }

    private static FreelancerSkill convertLongToFreelancerSkill(Long skillId) {
        return FreelancerSkill.getById(skillId);
    }

    private static Boolean convertWillRecommendFromUgc(TDirectReview.ERecommendationType recommendationType) {
        return recommendationType.equals(TDirectReview.ERecommendationType.WILL_RECOMMEND);
    }

    private static ZonedDateTime convertZonedDateTimeFromTimestamp(Timestamp timestamp) {
        if (StringUtils.isBlank(timestamp.toString())) {
            return null;
        }
        Instant instant = Instant.ofEpochSecond(timestamp.getSeconds(), timestamp.getNanos());
        return ZonedDateTime.ofInstant(instant, MSK);
    }

    private static Timestamp convertZonedDateTimeToTimestamp(ZonedDateTime dateTime) {
        if (dateTime == null) {
            return Timestamp.newBuilder().build();
        }
        return Timestamp.newBuilder()
                .setSeconds(dateTime.toEpochSecond())
                .setNanos(dateTime.getNano()).build();
    }

    private static TDirectReview.ERecommendationType convertWillRecommendToUgc(Boolean willRecommend) {
        if (willRecommend == null) {
            return TDirectReview.ERecommendationType.NONE;
        }
        return willRecommend
                ? TDirectReview.ERecommendationType.WILL_RECOMMEND
                : TDirectReview.ERecommendationType.NOT_RECOMMEND;
    }
}
