package ru.yandex.direct.core.entity.freelancer.service.validation;

import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.freelancer.model.Freelancer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerBase;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.rbac.PpcRbac;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notInSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.unconditional;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;

/**
 * Валидатор для добавляемого фрилансера.
 * <p>
 * Проверяем, что:
 * <ul>
 * <li>роль у соответствующего клиента {@code CLIENT}</li>
 * <li>конвертируемый клиент ещё не является фрилансером</li>
 * </ul>.
 */
@Service
public class AddFreelancerValidationService {

    private final PpcRbac ppcRbac;
    private final FreelancerService freelancerService;
    private final FreelancerCardValidator freelancerCardValidator;
    private final ClientService clientService;

    public AddFreelancerValidationService(
            PpcRbac ppcRbac,
            FreelancerService freelancerService,
            FreelancerCardValidator freelancerCardValidator,
            ClientService clientService) {
        this.ppcRbac = ppcRbac;
        this.freelancerService = freelancerService;
        this.freelancerCardValidator = freelancerCardValidator;
        this.clientService = clientService;
    }

    public ValidationResult<List<Freelancer>, Defect> validate(List<Freelancer> addingFreelancers) {
        List<Long> freelancerLongIds = StreamEx.of(addingFreelancers)
                .map(Freelancer::getId)
                .nonNull()
                .toList();
        Set<Long> existingFreelancerIds =
                listToSet(freelancerService.getFreelancers(freelancerLongIds), FreelancerBase::getId);

        List<ClientId> freelancerIds = mapList(freelancerLongIds, ClientId::fromLong);
        Map<ClientId, RbacRole> clientsRoles = ppcRbac.getClientsRoles(freelancerIds);
        Map<ClientId, Currency> currenciesByClientId = clientService.massGetWorkCurrency(freelancerIds);

        ListValidationBuilder<Freelancer, Defect> lvb = ListValidationBuilder.of(addingFreelancers);
        lvb.check(eachNotNull())
                .checkEachBy(certificatesNotNull())
                .checkEachBy(cardIsValid())
                .checkEachBy(fioIsValid())
                .checkEachBy(clientIsNotFreelancer(existingFreelancerIds))
                .checkEachBy(clientHasClientRole(clientsRoles), When.isValid())
                .checkEachBy(clientHasValidCurrency(currenciesByClientId), When.isValid());
        return lvb.getResult();
    }

    private Validator<Freelancer, Defect> clientHasValidCurrency(Map<ClientId, Currency> currenciesByClientId) {
        return freelancer -> {
            ModelItemValidationBuilder<Freelancer> vb = ModelItemValidationBuilder.of(freelancer);
            if (freelancer == null) {
                return vb.getResult();
            }
            Long id = freelancer.getId();
            Currency currency = currenciesByClientId.get(ClientId.fromLong(id));
            boolean currencyIsAbsent = currency == null || CurrencyCode.YND_FIXED.equals(currency.getCode());
            vb.item(Freelancer.ID)
                    .check(unconditional(FreelancerDefects.clientCurrencyIsNotConverted()),
                            When.isTrue(currencyIsAbsent));
            return vb.getResult();
        };
    }

    private Validator<Freelancer, Defect> clientHasClientRole(Map<ClientId, RbacRole> clientsRoles) {
        return freelancer -> {
            ModelItemValidationBuilder<Freelancer> vb = ModelItemValidationBuilder.of(freelancer);
            if (freelancer == null) {
                return vb.getResult();
            }
            Long id = freelancer.getId();
            RbacRole clientRole = clientsRoles.getOrDefault(ClientId.fromLong(id), RbacRole.EMPTY);

            boolean invalidRole = clientRole != RbacRole.CLIENT;
            vb.item(Freelancer.ID)
                    .check(notNull())
                    .check(unconditional(FreelancerDefects.clientHasWrongRole()), When.isTrue(invalidRole));
            return vb.getResult();
        };
    }

    private Validator<Freelancer, Defect> clientIsNotFreelancer(Set<Long> existingFreelancerIds) {
        return freelancer -> {
            ModelItemValidationBuilder<Freelancer> vb = ModelItemValidationBuilder.of(freelancer);
            if (freelancer == null) {
                return vb.getResult();
            }
            vb.item(Freelancer.ID)
                    .check(notInSet(existingFreelancerIds), FreelancerDefects.clientIsAlreadyFreelancer());
            return vb.getResult();
        };
    }

    private Validator<Freelancer, Defect> cardIsValid() {
        return freelancer -> {
            ModelItemValidationBuilder<Freelancer> vb = ModelItemValidationBuilder.of(freelancer);
            if (freelancer == null) {
                return vb.getResult();
            }
            vb.item(freelancer.getCard(), "card")
                    .checkBy(freelancerCardValidator);
            return vb.getResult();
        };
    }

    // Если у пользователя нет сертификатов, то должен быть пустой список, а не null.
    private Validator<Freelancer, Defect> certificatesNotNull() {
        return freelancer -> {
            ModelItemValidationBuilder<Freelancer> vb = ModelItemValidationBuilder.of(freelancer);
            if (freelancer == null) {
                return vb.getResult();
            }
            vb.item(Freelancer.CERTIFICATES)
                    .check(notNull());
            return vb.getResult();
        };
    }

    private Validator<Freelancer, Defect> fioIsValid() {
        return freelancer -> {
            ModelItemValidationBuilder<Freelancer> vb = ModelItemValidationBuilder.of(freelancer);
            if (freelancer == null) {
                return vb.getResult();
            }
            vb.item(Freelancer.FIRST_NAME)
                    .check(notNull())
                    .check(notBlank(), When.isValid());
            vb.item(Freelancer.SECOND_NAME)
                    .check(notNull())
                    .check(notBlank(), When.isValid());
            return vb.getResult();
        };
    }

}
