package ru.yandex.direct.core.entity.freelancer.service.validation;

import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.freelancer.model.FreelancerCard;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerContacts;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.ContactsConstraints.validIcq;
import static ru.yandex.direct.validation.constraint.ContactsConstraints.validPhone;
import static ru.yandex.direct.validation.constraint.ContactsConstraints.validSkype;
import static ru.yandex.direct.validation.constraint.ContactsConstraints.validTelegram;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.validEmail;
import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;

@Component
public class FreelancerCardValidator implements Validator<FreelancerCard, Defect> {

    static final int MAX_BRIEF_INFO_LENGTH = 10000;

    @Override
    public ValidationResult<FreelancerCard, Defect> apply(FreelancerCard freelancerCard) {
        ModelItemValidationBuilder<FreelancerCard> vb = ModelItemValidationBuilder.of(freelancerCard);
        vb.item(FreelancerCard.FREELANCER_ID)
                .check(notNull());
        vb.item(FreelancerCard.BRIEF_INFO)
                .check(maxStringLength(MAX_BRIEF_INFO_LENGTH), When.isValid());
        // emails и phone должны присутствовать
        ModelItemValidationBuilder<FreelancerContacts> contactsVb =
                vb.modelItem(FreelancerCard.CONTACTS);
        contactsVb.check(notNull());
        if (!contactsVb.getResult().hasAnyErrors()) {
            contactsVb.item(FreelancerContacts.PHONE)
                    .check(notNull())
                    .check(notBlank())
                    .check(validPhone());
            contactsVb.item(FreelancerContacts.EMAIL)
                    .check(notNull())
                    .check(notBlank())
                    .check(validEmail(), When.isValid());
            contactsVb.item(FreelancerContacts.ICQ)
                    .check(notBlank())
                    .check(validIcq());
            contactsVb.item(FreelancerContacts.SITE_URL)
                    .check(notBlank())
                    .check(validHref());
            contactsVb.item(FreelancerContacts.TELEGRAM)
                    .check(notBlank())
                    .check(validTelegram());
            contactsVb.item(FreelancerContacts.WHATS_APP)
                    .check(notBlank())
                    .check(validPhone());
            contactsVb.item(FreelancerContacts.SKYPE)
                    .check(notBlank())
                    .check(validSkype());
            contactsVb.item(FreelancerContacts.VIBER)
                    .check(notBlank())
                    .check(validPhone());
        }
        return vb.getResult();
    }

}
