package ru.yandex.direct.core.entity.freelancer.service.validation;

import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.freelancer.container.FreelancerProjectQueryContainer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerFeedback;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerProject;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerProjectStatus;
import ru.yandex.direct.core.entity.freelancer.repository.FreelancerProjectRepository;
import ru.yandex.direct.core.entity.freelancer.service.utils.FreelancerFeedbackConverter;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.ugcdb.client.UgcDbClient;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;
import ru.yandex.kernel.ugc.protos.direct.TDirectReview;

import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.feedbackAlreadyExist;
import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.operatorHasNoProjectWithFreelancer;
import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.operatorHasWrongRole;
import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.operatorIsNotChief;

@Service
@ParametersAreNonnullByDefault
public class FreelancerFeedbackValidationService {

    private final FreelancerProjectRepository freelancerProjectRepository;
    private final ShardHelper shardHelper;
    private final FreelancerFeedbackValidator feedbackValidator;
    private final ClientService clientService;
    private final UgcDbClient ugcDbClient;

    @Autowired
    public FreelancerFeedbackValidationService(
            FreelancerProjectRepository freelancerProjectRepository,
            ShardHelper shardHelper,
            FreelancerFeedbackValidator feedbackValidator,
            ClientService clientService, UgcDbClient ugcDbClient) {
        this.freelancerProjectRepository = freelancerProjectRepository;
        this.shardHelper = shardHelper;
        this.feedbackValidator = feedbackValidator;
        this.clientService = clientService;
        this.ugcDbClient = ugcDbClient;
    }

    public ValidationResult<FreelancerFeedback, Defect> validateAddFeedbackForFreelancer(ClientId clientId,
                                                                                         FreelancerFeedback feedback) {
        ModelItemValidationBuilder<FreelancerFeedback> vb = ModelItemValidationBuilder.of(feedback);

        vb.checkBy(feedbackValidator);

        // Проверяем, что фрилансер не пишет отзыв на себя
        vb.item(FreelancerFeedback.FREELANCER_ID)
                .check(operatorIsNotFreelancer(clientId.asLong()), When.isValid());

        // Проверяем, что автор является главным представителем клиента
        vb.item(FreelancerFeedback.AUTHOR_UID)
                .checkBy(operatorHasRightsToAddFeedback(clientId), When.isValid());

        // Проверяем, что у автора отзыва и фрилансера было сотрудничество
        vb.checkBy(hasCommonProjects(clientId.asLong()), When.isValid());

        // Проверяем, что этот автор еще не оставлял отзыв на данного клиента
        vb.checkBy(noFeedbackFromThisAuthor(), When.isValid());

        return vb.getResult();
    }

    public ValidationResult<FreelancerFeedback, Defect> validateUpdateFreelancerFeedback(Long operatorUid,
                                                                                         FreelancerFeedback feedback) {
        ModelItemValidationBuilder<FreelancerFeedback> vb = ModelItemValidationBuilder.of(feedback);

        vb.checkBy(feedbackValidator);

        // Проверяем, что оператор является автором модифицируемого отзыва
        vb.item(feedback.getAuthorUid(), "authorUid")
                .check(operatorIsFeedbackAuthor(operatorUid), When.isValid());

        return vb.getResult();
    }

    public ValidationResult<FreelancerFeedback, Defect> validateDeleteFreelancerFeedback(Long operatorUid,
                                                                                         FreelancerFeedback feedback) {
        ModelItemValidationBuilder<FreelancerFeedback> vb = ModelItemValidationBuilder.of(feedback);

        // Проверяем, что оператор является автором удаляемого отзыва
        vb.item(feedback.getAuthorUid(), "authorUid")
                .check(operatorIsFeedbackAuthor(operatorUid), When.isValid());

        return vb.getResult();
    }

    private Validator<FreelancerFeedback, Defect> hasCommonProjects(Long clientId) {
        return feedback -> {
            ItemValidationBuilder<FreelancerFeedback, Defect> vb = ItemValidationBuilder.of(feedback);
            int shard = shardHelper.getShardByClientId(ClientId.fromLong(clientId));
            FreelancerProjectQueryContainer queryContainer = FreelancerProjectQueryContainer.builder()
                    .withClientIds(clientId)
                    .withFreelancerIds(feedback.getFreelancerId())
                    .withStatuses(
                            FreelancerProjectStatus.INPROGRESS,
                            FreelancerProjectStatus.CANCELLEDBYCLIENT,
                            FreelancerProjectStatus.CANCELLEDBYFREELANCER)
                    .build();
            List<FreelancerProject> existingProjects =
                    freelancerProjectRepository.get(shard, queryContainer, LimitOffset.maxLimited()).stream()
                            .filter(project -> project.getStartedTime() != null)
                            .collect(Collectors.toList());
            vb.check(Constraint.fromPredicate(c -> !existingProjects.isEmpty(),
                    operatorHasNoProjectWithFreelancer()));
            return vb.getResult();
        };
    }

    private Validator<FreelancerFeedback, Defect> noFeedbackFromThisAuthor() {
        return feedback -> {
            ItemValidationBuilder<FreelancerFeedback, Defect> vb = ItemValidationBuilder.of(feedback);
            List<TDirectReview> reviewList = ugcDbClient.getFeedbackList(feedback.getFreelancerId());
            List<FreelancerFeedback> freelancerFeedbackList =
                    FreelancerFeedbackConverter.convertFreelancerFeedbackFromUgcDb(reviewList);
            vb.check(Constraint.fromPredicate(c ->
                            freelancerFeedbackList.stream().noneMatch(f -> f.getAuthorUid().equals(feedback.getAuthorUid())),
                    feedbackAlreadyExist()));
            return vb.getResult();
        };
    }

    private Validator<Long, Defect> operatorHasRightsToAddFeedback(ClientId clientId) {
        return operatorUid -> {
            ItemValidationBuilder<Long, Defect> vb = ItemValidationBuilder.of(operatorUid);
            Long chiefUid = clientService.getClient(clientId).getChiefUid();
            vb.check(Constraint.fromPredicate(c -> c.equals(chiefUid), operatorIsNotChief()));
            return vb.getResult();
        };
    }

    private static Constraint<Long, Defect> operatorIsNotFreelancer(Long clientId) {
        return Constraint.fromPredicate(freelancerId -> !freelancerId.equals(clientId), operatorHasWrongRole());
    }

    private static Constraint<Long, Defect> operatorIsFeedbackAuthor(Long operatorUid) {
        return Constraint
                .fromPredicate(feedbackAuthorUid -> feedbackAuthorUid.equals(operatorUid), operatorHasWrongRole());
    }

}
