package ru.yandex.direct.core.entity.freelancer.service.validation;

import java.util.Comparator;
import java.util.List;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.freelancer.model.FreelancerBase;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerSkill;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerSkillOffer;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Service
public class FreelancerSkillOffersValidationService {
    private final FreelancerService freelancerService;

    public FreelancerSkillOffersValidationService(
            FreelancerService freelancerService) {
        this.freelancerService = freelancerService;
    }

    public ValidationResult<List<FreelancerSkillOffer>, Defect> validate(
            List<FreelancerSkillOffer> freelancerSkillOffers) {
        List<Long> freelancerIds = StreamEx.of(freelancerSkillOffers)
                .map(FreelancerSkillOffer::getFreelancerId)
                .nonNull()
                .toList();
        Set<Long> existingFreelancerIds =
                listToSet(freelancerService.getFreelancers(freelancerIds), FreelancerBase::getId);
        ListValidationBuilder<FreelancerSkillOffer, Defect> lvb = ListValidationBuilder.of(freelancerSkillOffers);
        lvb.check(eachNotNull())
                .checkEachBy(isFreelancer(existingFreelancerIds), When.isValid())
                .checkEachBy(existSkill(), When.isValid())
                .checkEach(unique(getFreelancerSkillOfferComparator()), When.isValid());
        return lvb.getResult();
    }

    private Validator<FreelancerSkillOffer, Defect> isFreelancer(Set<Long> existingFreelancerIds) {
        return skillOffer -> {
            ModelItemValidationBuilder<FreelancerSkillOffer> vb = ModelItemValidationBuilder.of(skillOffer);
            vb.item(FreelancerSkillOffer.FREELANCER_ID)
                    .check(notNull())
                    .check(inSet(existingFreelancerIds), FreelancerDefects.mustBeFreelancer(), When.isValid());
            return vb.getResult();
        };
    }

    private Validator<FreelancerSkillOffer, Defect> existSkill() {
        return skillOffer -> {
            ModelItemValidationBuilder<FreelancerSkillOffer> vb = ModelItemValidationBuilder.of(skillOffer);
            vb.item(FreelancerSkillOffer.SKILL_ID)
                    .check(notNull())
                    .check(skillExists(), When.isValid());
            return vb.getResult();
        };
    }

    /**
     * {@link Constraint} для проверки ID услуги на существование
     */
    private Constraint<Long, Defect> skillExists() {
        return Constraint
                .fromPredicate(FreelancerSkill::isActiveSkillId,
                        FreelancerDefects.skillDoesNotExist());
    }

    private Comparator<FreelancerSkillOffer> getFreelancerSkillOfferComparator() {
        return Comparator.comparing(FreelancerSkillOffer::getFreelancerId)
                .thenComparing(FreelancerSkillOffer::getSkillId);
    }
}
