package ru.yandex.direct.core.entity.freelancer.service.validation;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.freelancer.container.FreelancerProjectQueryContainer;
import ru.yandex.direct.core.entity.freelancer.model.Freelancer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerProject;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerProjectIdentity;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerProjectStatus;
import ru.yandex.direct.core.entity.freelancer.repository.FreelancerProjectRepository;
import ru.yandex.direct.core.entity.freelancer.repository.FreelancerRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.freelancer.model.FreelancerProjectStatus.CANCELLEDBYCLIENT;
import static ru.yandex.direct.core.entity.freelancer.model.FreelancerProjectStatus.CANCELLEDBYFREELANCER;
import static ru.yandex.direct.core.entity.freelancer.model.FreelancerProjectStatus.INPROGRESS;
import static ru.yandex.direct.core.entity.freelancer.model.FreelancerProjectStatus.NEW;
import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.agencyClientCantRequestService;
import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.clientIsAlreadyFreelancer;
import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.mustBeClient;
import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.mustBeFreelancer;
import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.projectsAreAlreadyExist;
import static ru.yandex.direct.core.entity.freelancer.service.validation.FreelancerDefects.socialClientsCantRequestService;
import static ru.yandex.direct.dbutil.sharding.ShardSupport.NO_SHARD;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;

@Service
@ParametersAreNonnullByDefault
public class FreelancerValidationService {
    private final ShardHelper shardHelper;
    private final FreelancerProjectRepository freelancerProjectRepository;
    private final FreelancerRepository freelancerRepository;
    private final FeatureService featureService;

    private static final ImmutableMap<FreelancerProjectStatus, List<FreelancerProjectStatus>>
            AVAILABLE_PROJECT_TRANSITIONS_BY_FREELANCER =
            ImmutableMap.<FreelancerProjectStatus, List<FreelancerProjectStatus>>builder()
                    .put(NEW, asList(INPROGRESS, CANCELLEDBYFREELANCER))
                    .put(INPROGRESS, singletonList(CANCELLEDBYFREELANCER))
                    .build();

    private static final ImmutableMap<FreelancerProjectStatus, List<FreelancerProjectStatus>>
            AVAILABLE_PROJECT_TRANSITIONS_BY_CLIENT =
            ImmutableMap.<FreelancerProjectStatus, List<FreelancerProjectStatus>>builder()
                    .put(NEW, singletonList(CANCELLEDBYCLIENT))
                    .put(INPROGRESS, singletonList(CANCELLEDBYCLIENT))
                    .build();

    @Autowired
    public FreelancerValidationService(ShardHelper shardHelper,
                                       FreelancerProjectRepository freelancerProjectRepository,
                                       FreelancerRepository freelancerRepository, FeatureService featureService) {
        this.shardHelper = shardHelper;
        this.freelancerProjectRepository = freelancerProjectRepository;
        this.freelancerRepository = freelancerRepository;
        this.featureService = featureService;
    }

    public ValidationResult<FreelancerProjectIdentity, Defect> validateProjectIdentity(
            FreelancerProjectIdentity projectIdentity) {
        ModelItemValidationBuilder<FreelancerProjectIdentity> vb = ModelItemValidationBuilder.of(projectIdentity);
        vb.item(FreelancerProjectIdentity.CLIENT_ID)
                .check(notNull())
                .check(validId());
        vb.item(FreelancerProjectIdentity.FREELANCER_ID)
                .check(notNull())
                .check(validId());
        vb.item(FreelancerProjectIdentity.ID)
                .check(notNull())
                .check(validId());
        return vb.getResult();
    }

    public ValidationResult<Void, Defect> validateProjectRequest(User client, Long freelancerId) {
        ItemValidationBuilder<Void, Defect> vb = ItemValidationBuilder.of(null);
        vb.item(client.getClientId().asLong(), "operator")
                .check(isClient(client.getRole()))
                .check(isNotAgencyClient(client), When.isValid())
                .check(isNotFreelancer(), When.isValid())
                .check(isNotSocialClient(client), When.isValid())
                .checkBy(hasNoOpenedProjects(), When.isValid());

        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }

        vb.item(freelancerId, "freelancerId")
                .check(validId())
                .check(isFreelancer(), When.isValid());
        return vb.getResult();
    }

    private Constraint<Long, Defect> isNotSocialClient(User client) {
        return Constraint.fromPredicate(c -> !featureService.isEnabledForClientId(client.getClientId(),
                FeatureName.SOCIAL_ADVERTISING), socialClientsCantRequestService());
    }

    private Constraint<Long, Defect> isNotFreelancer() {
        return Constraint.fromPredicate(clientId -> {
            int shard = shardHelper.getShardByClientId(ClientId.fromLong(clientId));
            if (shard == NO_SHARD) {
                return false;
            }
            List<Freelancer> freelancers = freelancerRepository.getByIds(shard, singletonList(clientId));
            return freelancers.size() == 0;
        }, clientIsAlreadyFreelancer());
    }

    private Constraint<Long, Defect> isClient(RbacRole operatorRole) {
        return Constraint.fromPredicate(c -> RbacRole.CLIENT.equals(operatorRole), mustBeClient());
    }

    private Constraint<Long, Defect> isNotAgencyClient(User client) {
        return Constraint.fromPredicate(c -> client.getAgencyClientId() == null, agencyClientCantRequestService());
    }

    private Validator<Long, Defect> hasNoOpenedProjects() {
        return id -> {
            ItemValidationBuilder<Long, Defect> vb = ItemValidationBuilder.of(id);
            int shard = shardHelper.getShardByClientId(ClientId.fromLong(id));
            FreelancerProjectQueryContainer queryContainer = FreelancerProjectQueryContainer.builder()
                    .withClientIds(id)
                    .withStatuses(FreelancerProjectStatus.NEW, FreelancerProjectStatus.INPROGRESS)
                    .build();
            List<FreelancerProject> existingProjects =
                    freelancerProjectRepository.get(shard, queryContainer, LimitOffset.maxLimited());
            vb.check(Constraint.fromPredicate(c -> existingProjects.isEmpty(),
                    projectsAreAlreadyExist(mapList(existingProjects, FreelancerProject::getId))));
            return vb.getResult();
        };
    }

    private Constraint<Long, Defect> isFreelancer() {
        return Constraint.fromPredicate(id -> {
            int shard = shardHelper.getShardByClientId(ClientId.fromLong(id));
            if (shard == NO_SHARD) {
                return false;
            }
            List<Freelancer> freelancers = freelancerRepository.getByIds(shard, singletonList(id));
            return freelancers.size() == 1;
        }, mustBeFreelancer());
    }

    public ValidationResult<Long, Defect> validateChangeProjectStatusByFreelancer(
            FreelancerProjectIdentity projectIdentity,
            FreelancerProjectStatus newStatus,
            Map<Long, FreelancerProject> existingProjects) {
        return validateStatusChange(projectIdentity, newStatus, existingProjects,
                AVAILABLE_PROJECT_TRANSITIONS_BY_FREELANCER);
    }

    public ValidationResult<Long, Defect> validateChangeProjectStatusByClient(
            FreelancerProjectIdentity projectIdentity,
            FreelancerProjectStatus newStatus,
            Map<Long, FreelancerProject> existingProjects) {
        return validateStatusChange(projectIdentity, newStatus, existingProjects,
                AVAILABLE_PROJECT_TRANSITIONS_BY_CLIENT);
    }

    private ValidationResult<Long, Defect> validateStatusChange(
            FreelancerProjectIdentity projectIdentity,
            FreelancerProjectStatus newStatus,
            Map<Long, FreelancerProject> existingProjects,
            Map<FreelancerProjectStatus, List<FreelancerProjectStatus>> transitionsMap) {
        FreelancerProject project = existingProjects.getOrDefault(projectIdentity.getId(), null);
        ItemValidationBuilder<Long, Defect> v = ItemValidationBuilder.of(projectIdentity.getId());
        v.check(Constraint.fromPredicateOfNullable(t -> project != null, FreelancerDefects.projectNotFound()));

        if (v.getResult().hasAnyErrors()) {
            return v.getResult();
        }

        v.check(Constraint.fromPredicate(
                id -> transitionsMap.getOrDefault(project.getStatus(), emptyList())
                        .contains(newStatus), FreelancerDefects.transitionIsNotAvailable(project.getStatus())));
        return v.getResult();
    }

}
