package ru.yandex.direct.core.entity.freelancer.service.validation;

import java.util.List;
import java.util.Set;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter;
import ru.yandex.direct.core.entity.freelancer.model.Freelancer;
import ru.yandex.direct.core.entity.freelancer.model.FreelancerBase;
import ru.yandex.direct.core.entity.freelancer.service.FreelancerService;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;

import static ru.yandex.direct.core.entity.freelancer.container.FreelancersQueryFilter.allFreelancersIncludingDisabled;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class UpdateFreelancerValidationService {

    private final ModelChangesValidationTool preValidationTool;
    private final FreelancerService freelancerService;
    private final FreelancerBaseValidator freelancerValidator;

    public UpdateFreelancerValidationService(
            FreelancerService freelancerService,
            FreelancerBaseValidator freelancerValidator) {
        this.freelancerService = freelancerService;
        this.freelancerValidator = freelancerValidator;
        this.preValidationTool = ModelChangesValidationTool.builder().build();
    }

    public ValidationResult<List<ModelChanges<FreelancerBase>>, Defect> preValidate(
            List<ModelChanges<FreelancerBase>> modelChangesList) {
        List<Long> freelancerIds = mapList(modelChangesList, ModelChanges::getId);
        FreelancersQueryFilter queryFilter = allFreelancersIncludingDisabled()
                .withFreelancerIds(freelancerIds)
                .build();
        List<Freelancer> existingFreelancers = freelancerService.getFreelancers(queryFilter);
        Set<Long> existingIds = listToSet(existingFreelancers, FreelancerBase::getId);

        return preValidationTool.validateModelChangesList(modelChangesList, existingIds);
    }

    public ValidationResult<List<FreelancerBase>, Defect> validate(
            ValidationResult<List<FreelancerBase>, Defect> validationResult) {
        ListValidationBuilder<FreelancerBase, Defect> lvb = new ListValidationBuilder<>(validationResult);

        lvb.checkEachBy(freelancerValidator, When.isValid());
        return lvb.getResult();
    }
}
