package ru.yandex.direct.core.entity.geo.service;

import java.util.Optional;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.geobasehelper.GeoBaseHelper;
import ru.yandex.direct.libs.laas.LaasClient;

import static ru.yandex.direct.common.util.HttpUtil.getGdprCookie;
import static ru.yandex.direct.common.util.HttpUtil.getIsGdprCookie;
import static ru.yandex.direct.common.util.HttpUtil.getRequest;
import static ru.yandex.direct.common.util.HttpUtil.getSecretRemoteIp;
import static ru.yandex.direct.common.util.HttpUtil.getYandexUid;
import static ru.yandex.direct.common.util.HttpUtil.getYpCookie;

/**
 * Сервис для определения текущего местоположения пользователя
 */
@Service
public class CurrentGeoService {

    private static final Logger logger = LoggerFactory.getLogger(CurrentGeoService.class);

    private final GeoBaseHelper geoBaseHelper;
    private final LaasClient laasClient;

    @Autowired
    public CurrentGeoService(GeoBaseHelper geoBaseHelper, LaasClient laasClient) {
        this.geoBaseHelper = geoBaseHelper;
        this.laasClient = laasClient;
    }

    public Optional<Long> getCurrentDirectRegionId() {
        return getCurrentRegionId().flatMap(geoBaseHelper::convertToDirectRegionId);
    }

    public Optional<Long> getCurrentCountryRegionId() {
        return getCurrentRegionId().map(geoBaseHelper::getCountryId).map(Integer::longValue);
    }

    public Optional<Long> getCurrentRegionId() {
        logger.info("Trying to fetch region id by request");

        var request = getRequest();
        return getRegionIdByCookie(request)
                .or(() -> getRegionIdByIp(request));
    }

    private Optional<Long> getRegionIdByCookie(HttpServletRequest request) {
        Optional<Long> regionIdByCookie =
                canUseYandexGidCookie(request) ? HttpUtil.getCurrentGeoRegionId(request) : Optional.empty();

        logger.info("Trying to fetch region id from LaaS");
        String userIp = getSecretRemoteIp(request);
        String yandexUid = getYandexUid(request).orElse(null);
        String ypCookie = getYpCookie(request).orElse(null);
        Optional<Long> regionIdFromLaas =
                laasClient.getRegionId(userIp, yandexUid, regionIdByCookie.orElse(null), ypCookie);
        if (regionIdFromLaas.isPresent()) {
            return regionIdFromLaas;
        } else {
            logger.info("Trying to fetch region id by region cookie");
            return regionIdByCookie;
        }
    }

    private boolean canUseYandexGidCookie(HttpServletRequest request) {
        Optional<Long> gdprCookie = getGdprCookie(request);
        Optional<Long> isGdprCookie = getIsGdprCookie(request);

        // https://wiki.yandex-team.ru/users/4ernyakov/upravljator-kukami/#sostojanijakukigdpr
        // Кука yandex_gid находится в категории "остальные"
        return (gdprCookie.isPresent() && (gdprCookie.get() == 0 || gdprCookie.get() == 3))
                || (isGdprCookie.isPresent() && isGdprCookie.get() == 0);
    }

    private Optional<Long> getRegionIdByIp(HttpServletRequest request) {
        logger.info("Trying to fetch region id by ip");

        String ip = getSecretRemoteIp(request);

        if (ip == null) {
            return Optional.empty();
        }

        return Optional.of(geoBaseHelper.getRegionIdByIp(ip));
    }
}
