package ru.yandex.direct.core.entity.goal.repository;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.retargeting.model.MetrikaCounterGoalType;
import ru.yandex.direct.grid.schema.yt.Tables;
import ru.yandex.direct.grid.schema.yt.tables.ConvMainAdGoals;
import ru.yandex.direct.metrika.client.model.response.CounterGoal;
import ru.yandex.direct.ytcomponents.service.MetrikaConversionAdGoalsDynContextProvider;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtMappingUtils;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;

import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;

@Repository
@ParametersAreNonnullByDefault
public class MetrikaConversionAdGoalsRepository {
    public static final ConvMainAdGoals AD_GOALS = Tables.CONV_MAIN_AD_GOALS.as("G");
    private static final Field<Long> GOAL_ID = aliased(AD_GOALS.GOAL_ID);
    private static final Field<String> GOAL_TYPE = aliased(AD_GOALS.GOAL_TYPE);
    private static final Field<Long> COUNTER_ID = aliased(AD_GOALS.COUNTER_ID);
    private static final Field<String> GOAL_NAME = aliased(AD_GOALS.NAME);
    private static final Field<BigDecimal> DEFAULT_PRICE = aliased(AD_GOALS.DEFAULT_PRICE);

    private final MetrikaConversionAdGoalsDynContextProvider dynContextProvider;

    @Autowired
    public MetrikaConversionAdGoalsRepository(MetrikaConversionAdGoalsDynContextProvider dynContextProvider) {
        this.dynContextProvider = dynContextProvider;
    }

    public Map<Long, MetrikaCounterGoalType> getMetrikaGoalTypes(Set<Long> goalIds) {
        var query = YtDSL.ytContext()
                .select(GOAL_ID, GOAL_TYPE)
                .from(AD_GOALS)
                .where(GOAL_ID.in(goalIds));

        List<YTreeMapNode> rows = dynContextProvider.getContext()
                .executeSelect(query)
                .getYTreeRows();

        return StreamEx.of(rows)
                .mapToEntry(
                        r -> r.getLong(GOAL_ID.getName()),
                        r -> toMetrikaCounterGoalType(r.getString(GOAL_TYPE.getName())))
                .toMap();
    }

    public Map<Integer, List<CounterGoal>> getCounterGoalsByIds(Collection<Integer> counterIds,
                                                                Collection<Long> goalIds) {
        var query = YtDSL.ytContext()
                .select(GOAL_ID, COUNTER_ID, GOAL_NAME, DEFAULT_PRICE)
                .from(AD_GOALS)
                .where(GOAL_ID.in(goalIds))
                .and(COUNTER_ID.in(counterIds));

        var rows = dynContextProvider.getContext()
                .executeSelect(query)
                .getYTreeRows();

        return StreamEx.of(rows)
                .mapToEntry(
                        row -> (int) row.getLong(COUNTER_ID.getName()),
                        row -> new CounterGoal().withId((int) row.getLong(GOAL_ID.getName()))
                                .withName(row.getStringO(GOAL_NAME.getName()).orElse(null))
                                .withDefaultPrice(BigDecimal.valueOf(row.getDouble(DEFAULT_PRICE.getName())))
                )
                .grouping();
    }

    private MetrikaCounterGoalType toMetrikaCounterGoalType(String goalType) {
        return YtMappingUtils.enumFromString(goalType, MetrikaCounterGoalType.class);
    }
}
