package ru.yandex.direct.core.entity.goal.service;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.bannercategoriesmultik.client.BannerCategoriesMultikClient;
import ru.yandex.direct.bannercategoriesmultik.client.model.BannerCategories;
import ru.yandex.direct.bannercategoriesmultik.client.model.BannerInfo;
import ru.yandex.direct.bannercategoriesmultik.client.model.CategoriesRequest;
import ru.yandex.direct.bannercategoriesmultik.client.model.CategoriesResponse;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.currency.service.CurrencyRateService;
import ru.yandex.direct.core.entity.goal.repository.ConversionPriceForecastRepository;
import ru.yandex.direct.core.entity.goal.repository.MetrikaConversionAdGoalsRepository;
import ru.yandex.direct.core.entity.goal.repository.PriceWithClicks;
import ru.yandex.direct.core.entity.retargeting.model.MetrikaCounterGoalType;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.richcontent.RichContentClient;
import ru.yandex.direct.richcontent.model.UrlInfo;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.currency.CurrencyCode.YND_FIXED;

@Service
@ParametersAreNonnullByDefault
public class ConversionPriceForecastService {
    private static final Logger logger = LoggerFactory.getLogger(ConversionPriceForecastService.class);

    private static final int MAX_CATEGORIES_NUM = 1;

    private final RichContentClient richContentClient;
    private final BannerCategoriesMultikClient bannerCategoriesMultikClient;
    private final CurrencyRateService currencyRateService;
    private final ConversionPriceForecastRepository conversionPriceForecastRepository;
    private final MetrikaConversionAdGoalsRepository metrikaConversionAdGoalsRepository;
    private final ClientService clientService;
    private final ConversionPriceForecastServiceGeoHelper geoHelper;

    @Autowired
    public ConversionPriceForecastService(RichContentClient richContentClient,
                                          BannerCategoriesMultikClient bannerCategoriesMultikClient,
                                          CurrencyRateService currencyRateService,
                                          ConversionPriceForecastRepository conversionPriceForecastRepository,
                                          MetrikaConversionAdGoalsRepository metrikaConversionAdGoalsRepository,
                                          ClientService clientService,
                                          ConversionPriceForecastServiceGeoHelper geoHelper) {
        this.richContentClient = richContentClient;
        this.bannerCategoriesMultikClient = bannerCategoriesMultikClient;
        this.currencyRateService = currencyRateService;
        this.conversionPriceForecastRepository = conversionPriceForecastRepository;
        this.metrikaConversionAdGoalsRepository = metrikaConversionAdGoalsRepository;
        this.clientService = clientService;
        this.geoHelper = geoHelper;
    }

    public Map<Long, CampaignConversionPriceForGoalsWithCategoryCpaSource> getRecommendedConversionPriceByGoalIds(ClientId clientId,
                                                                                                                  Set<Long> goalIds,
                                                                                                                  Map<Long, String> urlByCampaignId) {
        CurrencyCode currencyCode = clientService.getWorkCurrency(clientId).getCode();
        Map<Long, List<Long>> geoByCampaignId = geoHelper.getCampRegionIds(clientId, urlByCampaignId);

        Map<Long, CampaignConversionPriceForGoalsWithCategoryCpaSource> recommendedPrice = new HashMap<>();
        urlByCampaignId.forEach((cid, url) -> recommendedPrice.put(
                cid,
                new CampaignConversionPriceForGoalsWithCategoryCpaSource(
                        getRecommendedPriceByGoalIds(goalIds, url, geoByCampaignId.get(cid), currencyCode))));

        return recommendedPrice;
    }

    public CampaignConversionPriceForGoalsWithCategoryCpaSource getRecommendedConversionPriceByGoalIdsForDraftCamp(ClientId clientId,
                                                                                                                   Set<Long> goalIds,
                                                                                                                   String url) {
        CurrencyCode currencyCode = clientService.getWorkCurrency(clientId).getCode();
        Long clientCountryId = clientService.getCountryRegionIdByClientIdStrict(clientId);

        return new CampaignConversionPriceForGoalsWithCategoryCpaSource(
                getRecommendedPriceByGoalIds(goalIds, url, List.of(clientCountryId), currencyCode));
    }

    private Map<Long, BigDecimal> getRecommendedPriceByGoalIds(Set<Long> goalIds,
                                                               String url,
                                                               List<Long> geo,
                                                               CurrencyCode currencyCode) {
        //TODO(ninazhevtyak) добавить проверку на доступность счетчиков с доменом url (DIRECT-153716)
        if (url.isEmpty()) {
            return emptyMap();
        }
        UrlInfo urlInfo = richContentClient.getUrlInfo(url);
        if (isInvalidUrlInfo(urlInfo)) {
            return emptyMap();
        }
        Long categoryId = getBannerCategoryId(urlInfo);

        Map<Long, MetrikaCounterGoalType> metrikaGoalTypes =
                metrikaConversionAdGoalsRepository.getMetrikaGoalTypes(goalIds);

        Map<Long, String> goalTypes = EntryStream.of(metrikaGoalTypes)
                .mapValues(goalType -> MetrikaCounterGoalType.toSource(goalType).getLiteral())
                .toMap();

        Map<String, List<PriceWithClicks>> meanPriceAndClicksByGoalTypes =
                conversionPriceForecastRepository.getPriceWithClicksByGoalTypes(
                        categoryId,
                        List.copyOf(goalTypes.values()),
                        geo);

        Map<Long, BigDecimal> recommendedPriceByGoalIds = new HashMap<>();
        goalTypes.forEach((goalId, goalType) -> {
            if (meanPriceAndClicksByGoalTypes.containsKey(goalType)) {
                try {
                    Double meanPriceInChips =
                            getWeightedMeanPrice(meanPriceAndClicksByGoalTypes.get(goalType));
                    recommendedPriceByGoalIds.put(
                            goalId,
                            getMoneyValueInClientCurrency(meanPriceInChips, currencyCode));
                } catch (IllegalStateException e) {
                    logger.error(e.getMessage());
                }
            }
        });

        return recommendedPriceByGoalIds;
    }

    private boolean isInvalidUrlInfo(UrlInfo urlInfo) {
        return urlInfo.getTitle() == null || urlInfo.getDescription() == null;
    }

    private Double getWeightedMeanPrice(List<PriceWithClicks> priceWithClicks) {
        double clicksNum = getAllClicksSum(priceWithClicks);
        double weightedPrice = getWeightedPrice(priceWithClicks);

        return weightedPrice / clicksNum;
    }

    private double getAllClicksSum(List<PriceWithClicks> priceWithClicks) {
        double clicksNum = 0;
        for (PriceWithClicks priceWithClicks1 : priceWithClicks) {
            double curRegionClicks = priceWithClicks1.getClicks();
            checkPositiveNumOfClicks(curRegionClicks);

            clicksNum += curRegionClicks;
        }

        return clicksNum;
    }

    private double getWeightedPrice(Collection<PriceWithClicks> priceWithClicksForRegions) {
        double weightedPrice = 0;
        for (PriceWithClicks priceWithClicks : priceWithClicksForRegions) {
            weightedPrice += priceWithClicks.getClicks() * priceWithClicks.getMeanPrice();
        }

        return weightedPrice;
    }

    private void checkPositiveNumOfClicks(double clicks) {
        if (clicks <= 0) {
            throw new IllegalStateException("Non-positive number of clicks " + clicks);
        }
    }


    private BigDecimal getMoneyValueInClientCurrency(Double meanPriceInChips, CurrencyCode currencyCode) {
        return currencyRateService.convertMoney(
                Money.valueOf(meanPriceInChips, YND_FIXED), currencyCode).roundToCentUp().bigDecimalValue();
    }

    private Long getBannerCategoryId(UrlInfo urlInfo) {
        CategoriesResponse response = bannerCategoriesMultikClient.getCategories(
                new CategoriesRequest(
                        List.of(new BannerInfo(urlInfo.getUrl(), urlInfo.getTitle(), urlInfo.getDescription())),
                        MAX_CATEGORIES_NUM));
        // для единственного баннера и модели возвращаем наиболее вероятную категорию (первая в списке)
        BannerCategories bannerResult = response.getBannerCategories().get(0);
        List<Long> bannerFirstModelCategories = bannerResult.getCategoriesByModelIds().get(0);
        return bannerFirstModelCategories.get(0);
    }
}
