package ru.yandex.direct.core.entity.goal.service;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;

import static java.lang.Math.min;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

@Component
@ParametersAreNonnullByDefault
public class ConversionPriceForecastServiceGeoHelper {
    private static final int HIGHEST_REGION_TYPE = 1;
    private static final int LOWEST_REGION_TYPE = 5; // минимальный уровень региона в таблице - Administrative area (5)

    private final AdGroupService adGroupService;
    private final ClientService clientService;
    private final GeoTreeFactory geoTreeFactory;

    public ConversionPriceForecastServiceGeoHelper(AdGroupService adGroupService,
                                                   ClientService clientService,
                                                   GeoTreeFactory geoTreeFactory) {
        this.adGroupService = adGroupService;
        this.clientService = clientService;
        this.geoTreeFactory = geoTreeFactory;
    }

    /**
     * Возвращает список регионов по id клиента и списку url, заданных на соответствующих кампаниях клиента.
     */
    public Map<Long, List<Long>> getCampRegionIds(ClientId clientId, Map<Long, String> urlByCampaignId) {
        Map<Long, List<Long>> defaultGeoByCampaignId =
                adGroupService.getDefaultGeoByCampaignId(clientId, urlByCampaignId.keySet());
        Map<Long, List<Long>> geo = removeMinusRegions(defaultGeoByCampaignId);

        geo.replaceAll((campaignId, regions) ->
                replaceWithClientCountryRegionIfAnyDefaultGeoTypeIsHigher(clientId, regions));
        geo.replaceAll(this::getLowestCommonGeoExceptGlobal);
        return geo;
    }

    private Map<Long, List<Long>> removeMinusRegions(Map<Long, List<Long>> defaultGeo) {
        return EntryStream.of(defaultGeo)
                .mapValues(this::filterMinusRegions)
                .toMap();
    }

    private List<Long> filterMinusRegions(List<Long> geos) {
        return filterList(geos, geo -> geo >= 0);
    }

    /**
     * Замещает дефолтные регионы клиента на его страну, если в в списке дефолтных гео есть хотя бы один регион,
     * тип которого выше континента.
     */
    private List<Long> replaceWithClientCountryRegionIfAnyDefaultGeoTypeIsHigher(ClientId clientId, List<Long> geo) {
        return geo.stream().anyMatch(rid -> getGeoTree().getRegion(rid).getType() < HIGHEST_REGION_TYPE)
                ? List.of(clientService.getCountryRegionIdByClientIdStrict(clientId))
                : geo;
    }

    /**
     * Возвращает ближайшего общего родителя для переданного списка регионов. Тип этого родителя должен быть не выше
     * уровня континента и не ниже уровня области. В случае, если нет общего родителя, возвращает список
     * регионов из исходного списка, где каждый регион при необходимости переведен в своего предка уровня 5
     * (Administrative area).
     */
    private List<Long> getLowestCommonGeoExceptGlobal(Long cid, List<Long> geo) {
        if (geo.size() == 1) {
            return geo;
        }

        geo = geo.stream()
                .map(this::getRegionId_OrAncestorOfLowestRegionType)
                .collect(toList());

        int geoHighestRegionType = Collections.min(geo.stream()
                .map(rid -> getGeoTree().getRegion(rid).getType())
                .collect(toList()));
        int curParentType = min(LOWEST_REGION_TYPE, geoHighestRegionType);

        Set<Long> parentsOfCurrentType = new HashSet<>();
        Set<Long> regionsOfPrevType = new HashSet<>(geo);
        while (curParentType >= HIGHEST_REGION_TYPE) {
            for (long regionId : regionsOfPrevType) {
                parentsOfCurrentType.add(getGeoTree().upRegionToType(regionId, curParentType));
            }

            if (parentsOfCurrentType.size() == 1) {
                return new ArrayList<>(parentsOfCurrentType);
            }

            curParentType--;
            regionsOfPrevType = Set.copyOf(parentsOfCurrentType);
            parentsOfCurrentType.clear();
        }

        return geo;
    }

    private long getRegionId_OrAncestorOfLowestRegionType(long regionId) {
        return getGeoTree().getRegion(regionId).getType() > LOWEST_REGION_TYPE
                ? getGeoTree().upRegionToType(regionId, LOWEST_REGION_TYPE)
                : regionId;
    }

    private GeoTree getGeoTree() {
        return geoTreeFactory.getRussianGeoTree();
    }
}
