package ru.yandex.direct.core.entity.grants.service;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.grants.model.Grants;
import ru.yandex.direct.core.entity.grants.service.validation.GrantsValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.rbac.ClientPerm;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.lang.Boolean.TRUE;
import static ru.yandex.direct.core.entity.grants.model.Grants.ALLOW_EDIT_CAMPAIGN;
import static ru.yandex.direct.core.entity.grants.model.Grants.ALLOW_IMPORT_XLS;
import static ru.yandex.direct.core.entity.grants.model.Grants.ALLOW_TRANSFER_MONEY;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Обновление разрешений, выданных субклиенту агенством
 */
public class GrantsUpdateOperation extends SimpleAbstractUpdateOperation<Grants, Long> {

    private final GrantsService grantsService;
    private final GrantsValidationService grantsValidationService;
    private final RbacService rbacService;
    private final ShardHelper shardHelper;
    private final ClientRepository clientRepository;
    private final long operatorUid;
    private final ClientId agencyId;

    public GrantsUpdateOperation(Applicability applicability, GrantsService grantsService,
                                 GrantsValidationService grantsValidationService,
                                 List<ModelChanges<Grants>> modelChanges,
                                 RbacService rbacService, ShardHelper shardHelper,
                                 ClientRepository clientRepository,
                                 long operatorUid, ClientId agencyId) {
        super(applicability, modelChanges, id -> new Grants().withId(id));
        this.grantsService = grantsService;
        this.grantsValidationService = grantsValidationService;
        this.rbacService = rbacService;
        this.shardHelper = shardHelper;
        this.clientRepository = clientRepository;
        this.operatorUid = operatorUid;
        this.agencyId = agencyId;
    }

    @Override
    protected ValidationResult<List<ModelChanges<Grants>>, Defect> validateModelChanges(
            List<ModelChanges<Grants>> modelChanges) {
        return new ValidationResult<>(modelChanges);
    }

    @Override
    protected Collection<Grants> getModels(Collection<Long> clientIds) {
        return grantsService.massGet(agencyId, mapList(clientIds, ClientId::fromLong));
    }

    @Override
    protected ValidationResult<List<Grants>, Defect> validateAppliedChanges(
            ValidationResult<List<Grants>, Defect> validationResult) {
        return new ListValidationBuilder<>(validationResult)
                .checkEachBy(grantsValidationService::validate)
                .getResult();
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<Grants>> validAppliedChanges) {
        validAppliedChanges.forEach(appliedChanges -> {
            ClientId clientId = ClientId.fromLong(appliedChanges.getModel().getId());

            if (appliedChanges.changed(ALLOW_EDIT_CAMPAIGN)
                    || appliedChanges.changed(ALLOW_IMPORT_XLS)
                    || appliedChanges.changed(ALLOW_TRANSFER_MONEY)) {
                Set<ClientPerm> perms = new HashSet<>();
                if (TRUE.equals(appliedChanges.getNewValue(ALLOW_EDIT_CAMPAIGN))) {
                    perms.add(ClientPerm.SUPER_SUBCLIENT);
                }
                if (TRUE.equals(appliedChanges.getNewValue(ALLOW_IMPORT_XLS))) {
                    perms.add(ClientPerm.XLS_IMPORT);
                }
                if (TRUE.equals(appliedChanges.getNewValue(ALLOW_TRANSFER_MONEY))) {
                    perms.add(ClientPerm.MONEY_TRANSFER);
                }
                clientRepository.setPerms(shardHelper.getShardByClientIdStrictly(clientId), clientId, perms);
                rbacService.clearCaches();
            }
        });
        return mapList(validAppliedChanges, a -> a.getModel().getId());
    }
}
