package ru.yandex.direct.core.entity.hrefparams.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.GROUP_PARAMS;

@Repository
@ParametersAreNonnullByDefault
public class HrefParamsRepository {
    private final DslContextProvider dslContextProvider;

    @Autowired
    HrefParamsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * @param shard     шард
     * @param adGroupId id группы
     * @return параметры шаблона на уровне группы или null, если для данного id записи о шаблонах нет
     */
    public String getHrefParamsOnAdGroup(int shard, Long adGroupId) {
        return getHrefParamsByAdGroupId(shard, List.of(adGroupId)).get(adGroupId);
    }

    /**
     * @param shard      шард
     * @param adGroupIds список id групп
     * @return словарь (id группы -> параметры шаблона на группе)
     */
    public Map<Long, String> getHrefParamsByAdGroupId(int shard, Collection<Long> adGroupIds) {
        return dslContextProvider.ppc(shard)
                .select(GROUP_PARAMS.PID, GROUP_PARAMS.HREF_PARAMS)
                .from(GROUP_PARAMS)
                .where(GROUP_PARAMS.PID.in(adGroupIds))
                .fetchMap(r -> r.getValue(GROUP_PARAMS.PID), r -> r.getValue(GROUP_PARAMS.HREF_PARAMS));
    }

    /**
     * @param shard      шард
     * @param campaignId id кампании
     * @return параметры шаблона на уровне кампании или null, если для данного id записи о шаблонах нет
     */
    public String getHrefParamsOnCampaign(int shard, Long campaignId) {
        return getHrefParamsByCampaignId(shard, List.of(campaignId)).get(campaignId);
    }

    /**
     * @param shard       шард
     * @param campaignIds список id кампаний
     * @return словарь (id кампании -> параметры шаблона на кампании)
     */
    public Map<Long, String> getHrefParamsByCampaignId(int shard, Collection<Long> campaignIds) {
        return dslContextProvider.ppc(shard)
                .select(CAMP_OPTIONS.CID, CAMP_OPTIONS.HREF_PARAMS)
                .from(CAMP_OPTIONS)
                .where(CAMP_OPTIONS.CID.in(campaignIds))
                .fetchMap(r -> r.getValue(CAMP_OPTIONS.CID), r -> r.getValue(CAMP_OPTIONS.HREF_PARAMS));
    }
}
