package ru.yandex.direct.core.entity.hrefparams.service;

import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.hrefparams.repository.HrefParamsRepository;
import ru.yandex.direct.utils.model.UrlParts;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.concat;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Service
public class HrefWithParamsBuildingService {

    private final HrefParamsRepository hrefParamsRepository;
    private final AdGroupRepository adGroupRepository;

    public HrefWithParamsBuildingService(HrefParamsRepository hrefParamsRepository,
                                         AdGroupRepository adGroupRepository) {

        this.hrefParamsRepository = hrefParamsRepository;
        this.adGroupRepository = adGroupRepository;
    }

    public static String buildHrefWithParams(@Nullable String bannerHref, String groupParams, String campaignParams) {
        if (bannerHref == null) {
            return null;
        }
        var trackingParams = isEmpty(groupParams) ? campaignParams : groupParams;
        return applyTrackingParams(UrlParts.fromUrl(bannerHref), UrlParts.parseParameters(trackingParams))
                .toUrl();
    }

    /**
     * Возвращает ссылку баннера вместе с шаблоном параметров.
     * Учитывает параметры в ссылке баннера, объединяя их с шаблоном, причем совпадающие ключи
     * в ссылке перетираются ключами шаблона.
     *
     * @param adGroupId  id группы
     * @param bannerHref текущая ссылка на баннере
     * @return url c объединенными параметрами
     */
    public String buildHrefWithParamsByAdGroupId(int shard, Long adGroupId, @Nullable String bannerHref) {
        if (bannerHref == null) {
            return null;
        }

        var campaignId = adGroupRepository.getCampaignIdByAdGroupId(shard, adGroupId);

        var trackingParams = selectTrackingParams(shard, campaignId, adGroupId);
        return applyTrackingParams(UrlParts.fromUrl(bannerHref), UrlParts.parseParameters(trackingParams))
                .toUrl();
    }

    /**
     * Метод, который из существующих шаблонов параметров на всех уровнях выбирает для подстановки в url
     * самый узкий шаблон.
     */
    private String selectTrackingParams(int shard, Long campaignId, Long adGroupId) {
        var paramsOnAdGroup = hrefParamsRepository.getHrefParamsOnAdGroup(shard, adGroupId);
        return isEmpty(paramsOnAdGroup)
                ? hrefParamsRepository.getHrefParamsOnCampaign(shard, campaignId)
                : paramsOnAdGroup;
    }

    public static UrlParts applyTrackingParams(UrlParts href, List<Pair<String, String>> trackingParams) {
        Set<String> trackingParamsKeySet = nvl(listToSet(trackingParams, Pair::getKey), emptySet());
        List<Pair<String, String>> filteredParamsOnHref = filterList(href.getParameters(),
                param -> !trackingParamsKeySet.contains(param.getKey()));

        List<Pair<String, String>> combinedParams = concat(
                nvl(filteredParamsOnHref, emptyList()),
                nvl(trackingParams, emptyList())
        );
        return href.toBuilder().withParameters(combinedParams).build();
    }
}
