package ru.yandex.direct.core.entity.hypergeo.operation;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.audience.client.YaAudienceClient;
import ru.yandex.direct.audience.client.exception.YaAudienceClientException;
import ru.yandex.direct.audience.client.model.SegmentResponse;
import ru.yandex.direct.audience.client.model.geosegment.YaAudienceGeoPoint;
import ru.yandex.direct.audience.client.model.geosegment.YaAudienceGeoSegmentType;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeoSegment;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeoSegmentDetails;
import ru.yandex.direct.core.entity.hypergeo.model.HyperPoint;
import ru.yandex.direct.core.util.HyperGeoUtils;
import ru.yandex.direct.geobasehelper.GeoBaseHelper;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static com.google.common.base.Preconditions.checkState;
import static java.lang.Math.cos;
import static java.lang.Math.toRadians;
import static ru.yandex.direct.core.util.HyperGeoUtils.convertGeoSegmentIdToGoalId;
import static ru.yandex.direct.regions.Region.DELETED_REGION_ID;
import static ru.yandex.direct.regions.Region.GLOBAL_REGION_ID;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class HyperGeoOperationsHelper {
    private static final Logger logger = LoggerFactory.getLogger(HyperGeoOperationsHelper.class);

    private static final double AVERAGE_EARTH_RADIUS_IN_METERS = 6371302;
    private static final double MERIDIAN_LENGTH_IN_METERS = AVERAGE_EARTH_RADIUS_IN_METERS * Math.PI;

    private final YaAudienceClient yaAudienceClient;
    private final GeoBaseHelper geoBaseHelper;
    private final GeoTreeFactory geoTreeFactory;

    @Autowired
    public HyperGeoOperationsHelper(YaAudienceClient yaAudienceClient,
                                    GeoBaseHelper geoBaseHelper,
                                    GeoTreeFactory geoTreeFactory) {
        this.yaAudienceClient = yaAudienceClient;
        this.geoBaseHelper = geoBaseHelper;
        this.geoTreeFactory = geoTreeFactory;
    }

    public void createGeoSegments(Collection<HyperGeoSegment> hyperGeoSegments, String login) {
        hyperGeoSegments.forEach(hyperGeoSegment -> {
            HyperGeoSegmentDetails hyperGeoSegmentDetails = hyperGeoSegment.getSegmentDetails();

            var yaGeoSegmentType = YaAudienceGeoSegmentType
                    .fromTypedValue(hyperGeoSegmentDetails.getGeoSegmentType().getTypedValue());

            Set<YaAudienceGeoPoint> yaAudienceGeoPoints = convertPoints(hyperGeoSegmentDetails.getPoints());
            SegmentResponse segmentResponse = yaAudienceClient.createGeoSegment(
                    login,
                    hyperGeoSegmentDetails.getSegmentName(),
                    hyperGeoSegmentDetails.getRadius(),
                    yaAudienceGeoPoints,
                    yaGeoSegmentType,
                    hyperGeoSegmentDetails.getPeriodLength(),
                    hyperGeoSegmentDetails.getTimesQuantity());

            Long hyperGeoSegmentId = convertGeoSegmentIdToGoalId(segmentResponse.getAudienceSegment().getId());
            hyperGeoSegment.withId(hyperGeoSegmentId);
        });
    }

    public static Set<YaAudienceGeoPoint> convertPoints(Collection<HyperPoint> hyperPoints) {
        return listToSet(hyperPoints,
                hyperPoint -> new YaAudienceGeoPoint(hyperPoint.getLatitude(), hyperPoint.getLongitude()));
    }

    /**
     * Удаляет сегменты в Я.Аудиториях.
     *
     * @param hyperGeoSegmentIds - goal_id сегментов на удаление
     * @return список id успешно удаленных сегментов или null, если таковых нет/запрос в Аудитории вернул ошибку
     */
    public List<Long> deleteGeoSegments(Collection<Long> hyperGeoSegmentIds) {
        var hyperGeoSegments = mapList(hyperGeoSegmentIds, HyperGeoUtils::convertGoalIdToGeoSegment);
        var deletedIdToSuccess = new HashMap<Long, Boolean>(hyperGeoSegments.size());
        for (var hyperGeoSegment : hyperGeoSegments) {
            try {
                var deletionResult = yaAudienceClient.deleteSegment(hyperGeoSegment);
                deletedIdToSuccess.put(hyperGeoSegment, deletionResult);
            } catch (YaAudienceClientException e) {
                logger.error("Exception while trying to delete " + hyperGeoSegment
                        + " hypergeo segment in Y.Audience ", e);
            }
        }

        var deletedIdsBySuccess = StreamEx.of(deletedIdToSuccess.entrySet())
                .groupingBy(Map.Entry::getValue, Collectors.mapping(Map.Entry::getKey, Collectors.toList()));
        if (!isEmpty(deletedIdsBySuccess.get(true))) {
            logger.info("Successfully deleted " + deletedIdsBySuccess.get(true)
                    + " hypergeo segments in Y.Audience");
        }
        if (!isEmpty(deletedIdsBySuccess.get(false))) {
            logger.info("Unsuccessfully tried to delete " + deletedIdsBySuccess.get(false)
                    + " hypergeo segments in Y.Audience");
        }
        return mapList(deletedIdsBySuccess.get(true), HyperGeoUtils::convertGeoSegmentIdToGoalId);
    }

    public void fillCoveringGeo(Collection<HyperGeoSegment> models) {
        models.forEach(this::fillCoveringGeo);
    }

    private void fillCoveringGeo(HyperGeoSegment hyperGeoSegment) {
        HyperGeoSegmentDetails hyperGeoSegmentDetails = hyperGeoSegment.getSegmentDetails();
        List<HyperPoint> hyperPoints = hyperGeoSegmentDetails.getPoints();

        checkState(hyperPoints.size() == 1, "multiple points not supported");
        HyperPoint center = hyperPoints.get(0);

        long coveringRegionId = findRegionIdCoveringCircle(center, hyperGeoSegmentDetails.getRadius());
        hyperGeoSegment.withCoveringGeo(List.of(coveringRegionId));
    }

    private long findRegionIdCoveringCircle(HyperPoint center, int radius) {
        double centerLatitude = center.getLatitude();
        double centerLongitude = center.getLongitude();
        double latitudeDelta = radius * 180 / MERIDIAN_LENGTH_IN_METERS;
        double longitudeDelta = radius * 180 / (MERIDIAN_LENGTH_IN_METERS * cos(toRadians(centerLatitude)));

        List<HyperPoint> boundingBoxPoints = StreamEx.of(centerLatitude - latitudeDelta, centerLatitude + latitudeDelta)
                .cross(centerLongitude - longitudeDelta, centerLongitude + longitudeDelta)
                .mapKeyValue((lat, lon) -> new HyperPoint()
                        .withLatitude(lat)
                        .withLongitude(lon))
                .toList();

        try (TraceProfile profile = Trace.current().profile("findRegionIdCoveringPoints")) {
            return findRegionIdCoveringPoints(boundingBoxPoints);
        }
    }

    private long findRegionIdCoveringPoints(Collection<HyperPoint> points) {
        Set<Long> pointsRegionIds = StreamEx.of(getRegionIdsOf(points))
                .filter(regionId -> !regionId.equals(DELETED_REGION_ID))
                .toSet();

        if (pointsRegionIds.isEmpty()) {
            return GLOBAL_REGION_ID;
        }

        List<List<Integer>> pointsParentRegionIds = getParentsRegionIds(pointsRegionIds);
        int minParentsSize = StreamEx.of(pointsParentRegionIds).map(List::size).reduce(Integer::min).get();

        for (int geoTreeDepth = minParentsSize - 1; geoTreeDepth >= 0; geoTreeDepth--) {
            final int currDepth = geoTreeDepth;
            Set<Integer> parentRegionIdsOnCurrDepth = StreamEx.of(pointsParentRegionIds)
                    .map(parentRegionIds -> parentRegionIds.get(currDepth))
                    .toSet();

            if (parentRegionIdsOnCurrDepth.size() != 1) {
                continue;
            }

            long candidateRegionId = parentRegionIdsOnCurrDepth.iterator().next();
            if (getGeoTree().hasRegion(candidateRegionId)) {
                return candidateRegionId;
            }
        }

        return GLOBAL_REGION_ID;
    }

    private List<Long> getRegionIdsOf(Collection<HyperPoint> points) {
        return StreamEx.of(points)
                .map(point -> geoBaseHelper.getRegionIdByCoordinates(point.getLatitude(), point.getLongitude()))
                .toList();
    }

    private List<List<Integer>> getParentsRegionIds(Collection<Long> regionIds) {
        return StreamEx.of(regionIds)
                .map(geoBaseHelper::getParentRegionIds)
                .peek(Collections::reverse)
                .toList();
    }

    private GeoTree getGeoTree() {
        return geoTreeFactory.getGlobalGeoTree();
    }
}
