package ru.yandex.direct.core.entity.hypergeo.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.jooq.Condition;
import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.TableField;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeo;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeoSegment;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeoSimple;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.core.entity.adgroup.repository.AdGroupMappings.geoFromDb;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_HYPERGEO_RETARGETINGS;
import static ru.yandex.direct.dbschema.ppc.Tables.HYPERGEO_SEGMENTS;
import static ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_CONDITIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_GOALS;
import static ru.yandex.direct.dbschema.ppc.enums.RetargetingConditionsRetargetingConditionsType.geo_segments;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class HyperGeoRepository {

    private final DslContextProvider dslContextProvider;

    @Autowired
    public HyperGeoRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    public void linkHyperGeosToAdGroups(Configuration config, Map<Long, Long> hyperGeoIdByAdGroupId) {
        if (hyperGeoIdByAdGroupId.isEmpty()) {
            return;
        }

        var step = config.dsl()
                .insertInto(ADGROUPS_HYPERGEO_RETARGETINGS)
                .columns(ADGROUPS_HYPERGEO_RETARGETINGS.PID, ADGROUPS_HYPERGEO_RETARGETINGS.RET_COND_ID);

        hyperGeoIdByAdGroupId.forEach(step::values);

        step.execute();
    }

    public void unlinkHyperGeosFromAdGroups(Configuration config, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return;
        }

        config.dsl()
                .deleteFrom(ADGROUPS_HYPERGEO_RETARGETINGS)
                .where(ADGROUPS_HYPERGEO_RETARGETINGS.PID.in(adGroupIds))
                .execute();
    }

    public Map<Long, List<Long>> getAdGroupIdsByHyperGeoId(int shard, Collection<Long> hyperGeoIds) {
        if (hyperGeoIds.isEmpty()) {
            return emptyMap();
        }

        return dslContextProvider.ppc(shard)
                .select(ADGROUPS_HYPERGEO_RETARGETINGS.PID, ADGROUPS_HYPERGEO_RETARGETINGS.RET_COND_ID)
                .from(ADGROUPS_HYPERGEO_RETARGETINGS)
                .where(ADGROUPS_HYPERGEO_RETARGETINGS.RET_COND_ID.in(hyperGeoIds))
                .fetchGroups(ADGROUPS_HYPERGEO_RETARGETINGS.RET_COND_ID, ADGROUPS_HYPERGEO_RETARGETINGS.PID);
    }

    public Set<Long> getHyperGeoIds(Configuration config, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return Set.of();
        }

        return config.dsl()
                .select(ADGROUPS_HYPERGEO_RETARGETINGS.RET_COND_ID)
                .from(ADGROUPS_HYPERGEO_RETARGETINGS)
                .where(ADGROUPS_HYPERGEO_RETARGETINGS.PID.in(adGroupIds))
                .fetchSet(ADGROUPS_HYPERGEO_RETARGETINGS.RET_COND_ID);
    }

    public Map<Long, HyperGeo> getHyperGeoByAdGroupId(int shard, ClientId clientId, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return emptyMap();
        }

        var hyperGeoRawByAdGroupId = dslContextProvider.ppc(shard)
                .select(ADGROUPS_HYPERGEO_RETARGETINGS.PID,
                        RETARGETING_CONDITIONS.RET_COND_ID,
                        RETARGETING_CONDITIONS.CONDITION_NAME,
                        HYPERGEO_SEGMENTS.GOAL_ID,
                        HYPERGEO_SEGMENTS.CLIENT_ID,
                        HYPERGEO_SEGMENTS.COVERING_GEO,
                        HYPERGEO_SEGMENTS.GEOSEGMENT_DETAILS)
                .from(HYPERGEO_SEGMENTS)
                .join(RETARGETING_GOALS).on(HYPERGEO_SEGMENTS.GOAL_ID.eq(RETARGETING_GOALS.GOAL_ID))
                .join(RETARGETING_CONDITIONS).on(RETARGETING_GOALS.RET_COND_ID.eq(RETARGETING_CONDITIONS.RET_COND_ID))
                .join(ADGROUPS_HYPERGEO_RETARGETINGS).on(RETARGETING_CONDITIONS.RET_COND_ID.eq(ADGROUPS_HYPERGEO_RETARGETINGS.RET_COND_ID))
                .where(ADGROUPS_HYPERGEO_RETARGETINGS.PID.in(adGroupIds))
                .and(RETARGETING_CONDITIONS.CLIENT_ID.eq(clientId.asLong()))
                .fetchGroups(new TableField[] { ADGROUPS_HYPERGEO_RETARGETINGS.PID,
                        RETARGETING_CONDITIONS.RET_COND_ID, RETARGETING_CONDITIONS.CONDITION_NAME });

        return EntryStream.of(hyperGeoRawByAdGroupId)
                .mapToValue((key, values) -> new HyperGeo()
                        .withId(key.get(RETARGETING_CONDITIONS.RET_COND_ID))
                        .withName(key.get(RETARGETING_CONDITIONS.CONDITION_NAME))
                        .withHyperGeoSegments(convert(values)))
                .mapKeys(key -> key.get(ADGROUPS_HYPERGEO_RETARGETINGS.PID))
                .toMap();
    }

    public Map<Long, HyperGeo> getHyperGeoById(int shard, ClientId clientId, Collection<Long> hyperGeoIds) {
        var condition = RETARGETING_CONDITIONS.CLIENT_ID.eq(clientId.asLong());
        return getHyperGeoById(shard, condition, hyperGeoIds);
    }

    private Map<Long, HyperGeo> getHyperGeoById(int shard, Condition condition, Collection<Long> hyperGeoIds) {
        if (hyperGeoIds.isEmpty()) {
            return emptyMap();
        }

        var hyperGeoRawById = dslContextProvider.ppc(shard)
                .select(RETARGETING_CONDITIONS.RET_COND_ID,
                        RETARGETING_CONDITIONS.CONDITION_NAME,
                        HYPERGEO_SEGMENTS.GOAL_ID,
                        HYPERGEO_SEGMENTS.CLIENT_ID,
                        HYPERGEO_SEGMENTS.COVERING_GEO,
                        HYPERGEO_SEGMENTS.GEOSEGMENT_DETAILS)
                .from(HYPERGEO_SEGMENTS)
                .join(RETARGETING_GOALS).on(HYPERGEO_SEGMENTS.GOAL_ID.eq(RETARGETING_GOALS.GOAL_ID))
                .join(RETARGETING_CONDITIONS).on(RETARGETING_GOALS.RET_COND_ID.eq(RETARGETING_CONDITIONS.RET_COND_ID))
                .where(RETARGETING_CONDITIONS.RET_COND_ID.in(hyperGeoIds))
                .and(condition)
                .fetchGroups(new TableField[] { RETARGETING_CONDITIONS.RET_COND_ID, RETARGETING_CONDITIONS.CONDITION_NAME });

        return EntryStream.of(hyperGeoRawById)
                .mapToValue((key, values) -> new HyperGeo()
                        .withId(key.get(RETARGETING_CONDITIONS.RET_COND_ID))
                        .withName(key.get(RETARGETING_CONDITIONS.CONDITION_NAME))
                        .withHyperGeoSegments(convert(values)))
                .mapKeys(key -> key.get(RETARGETING_CONDITIONS.RET_COND_ID))
                .toMap();
    }

    private List<HyperGeoSegment> convert(Result<? extends Record> records) {
        return mapList(records, this::convert);
    }

    private HyperGeoSegment convert(Record record) {

        var hyperGeoSegmentDetails =
                HyperGeoSegmentMappings.geoSegmentDetailsFromJson(record.get(HYPERGEO_SEGMENTS.GEOSEGMENT_DETAILS));

        return new HyperGeoSegment()
                .withId(record.get(HYPERGEO_SEGMENTS.GOAL_ID))
                .withClientId(record.get(HYPERGEO_SEGMENTS.CLIENT_ID))
                .withCoveringGeo(geoFromDb(record.get(HYPERGEO_SEGMENTS.COVERING_GEO)))
                .withSegmentDetails(hyperGeoSegmentDetails);
    }

    public Map<Long, HyperGeoSimple> getHyperGeoSimpleById(int shard, ClientId clientId, Collection<Long> hyperGeoIds) {
        if (hyperGeoIds.isEmpty()) {
            return emptyMap();
        }

        var hyperGeoRawById = dslContextProvider.ppc(shard)
                .select(RETARGETING_CONDITIONS.RET_COND_ID,
                        RETARGETING_CONDITIONS.CONDITION_NAME,
                        RETARGETING_GOALS.GOAL_ID)
                .from(RETARGETING_GOALS)
                .join(RETARGETING_CONDITIONS).on(RETARGETING_GOALS.RET_COND_ID.eq(RETARGETING_CONDITIONS.RET_COND_ID))
                .where(RETARGETING_CONDITIONS.RET_COND_ID.in(hyperGeoIds))
                .and(RETARGETING_CONDITIONS.CLIENT_ID.eq(clientId.asLong()))
                .fetchGroups(new TableField[] { RETARGETING_CONDITIONS.RET_COND_ID, RETARGETING_CONDITIONS.CONDITION_NAME });

        return EntryStream.of(hyperGeoRawById)
                .mapToValue((key, values) -> new HyperGeoSimple()
                        .withId(key.get(RETARGETING_CONDITIONS.RET_COND_ID))
                        .withName(key.get(RETARGETING_CONDITIONS.CONDITION_NAME))
                        .withHyperGeoSegmentIds(mapList(values, row -> row.get(RETARGETING_GOALS.GOAL_ID))))
                .mapKeys(key -> key.get(RETARGETING_CONDITIONS.RET_COND_ID))
                .toMap();
    }

    /**
     * Возвращает те id гипер гео, которые не привязаны ни к одной группе
     * (т.е. нет записей в adgroups_hypergeo_retargetings и, соответственно, в adgroup_additional_targetings).
     *
     * @param dslContext  - контекст транзакции
     * @param clientId    - id клиента
     * @param hyperGeoIds - id гипер гео, среди которых ищем неиспользуемые
     * @return Список id гипер гео
     */
    public Set<Long> getUnusedHyperGeoIds(DSLContext dslContext,
                                          @Nullable ClientId clientId,
                                          Collection<Long> hyperGeoIds) {
        if (isEmpty(hyperGeoIds)) {
            return Set.of();
        }

        Condition condition = RETARGETING_CONDITIONS.RET_COND_ID.in(hyperGeoIds);
        if (clientId != null) {
            condition = condition.and(RETARGETING_CONDITIONS.CLIENT_ID.eq(clientId.asLong()));
        }

        return getUnusedHyperGeoIds(dslContext, condition, null);
    }

    /**
     * ИСПОЛЬЗУЕТСЯ ТОЛЬКО В ДЖОБЕ И ВАНШОТЕ DeleteUnusedHyperGeos*.
     * Тесты на метод смотреть в DeleteUnusedHyperGeosOneshotTest.
     *
     * Пример запроса - https://yql.yandex-team.ru/Operations/Yk2i1pfFt9VV_rkDAr-uT72XcAu5_gLsjeuLewpBa-c=
     */
    @QueryWithoutIndex("Получение id неиспользуемых гипер гео (условий ретаргетинга) для их дальнейшего удаления;" +
            " редкая операция - используется только в ежедневной джобе и в ваншоте")
    public Set<Long> getUnusedHyperGeoIds(int shard, int limit) {
        return getUnusedHyperGeoIds(dslContextProvider.ppc(shard), DSL.noCondition(), limit);
    }

    private Set<Long> getUnusedHyperGeoIds(DSLContext dslContext,
                                           Condition condition,
                                           @Nullable Integer limit) {
        condition = condition
                .and(RETARGETING_CONDITIONS.RETARGETING_CONDITIONS_TYPE.eq(geo_segments))
                .and(RETARGETING_CONDITIONS.IS_DELETED.eq(RepositoryUtils.FALSE))
                .and(ADGROUPS_HYPERGEO_RETARGETINGS.RET_COND_ID.isNull());

        var selectConditionStep = dslContext
                .select(RETARGETING_CONDITIONS.RET_COND_ID)
                .from(RETARGETING_CONDITIONS)
                .leftJoin(ADGROUPS_HYPERGEO_RETARGETINGS)
                .on(ADGROUPS_HYPERGEO_RETARGETINGS.RET_COND_ID.eq(RETARGETING_CONDITIONS.RET_COND_ID))
                .where(condition);

        if (limit != null) {
            selectConditionStep
                    .limit(limit);
        }

        return selectConditionStep
                .fetchSet(RETARGETING_CONDITIONS.RET_COND_ID);
    }

    /**
     * ИСПОЛЬЗУЕТСЯ ТОЛЬКО ДЛЯ ЛОГИРОВАНИЯ В ДЖОБЕ И ВАНШОТЕ DeleteUnusedHyperGeos*.
     */
    public Collection<HyperGeo> getHyperGeosToLog(int shard, Collection<Long> hyperGeoIds) {
        var condition = RETARGETING_CONDITIONS.IS_DELETED.eq(RepositoryUtils.FALSE);
        return getHyperGeoById(shard, condition, hyperGeoIds).values();
    }
}
