package ru.yandex.direct.core.entity.hypergeo.repository;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupMappings;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeoSegment;
import ru.yandex.direct.dbschema.ppc.tables.records.HypergeoSegmentsRecord;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.dbschema.ppc.Tables.HYPERGEO_SEGMENTS;
import static ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_CONDITIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.RETARGETING_GOALS;
import static ru.yandex.direct.dbschema.ppc.enums.RetargetingConditionsRetargetingConditionsType.geo_segments;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;

@Repository
@ParametersAreNonnullByDefault
public class HyperGeoSegmentRepository {
    private static final Logger logger = LoggerFactory.getLogger(HyperGeoSegmentRepository.class);

    private final DslContextProvider dslContextProvider;

    private final JooqMapperWithSupplier<HyperGeoSegment> hyperGeoSegmentMapper;

    @Autowired
    public HyperGeoSegmentRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        hyperGeoSegmentMapper = JooqMapperWithSupplierBuilder.builder(HyperGeoSegment::new)
                .map(property(HyperGeoSegment.ID, HYPERGEO_SEGMENTS.GOAL_ID))
                .map(property(HyperGeoSegment.CLIENT_ID, HYPERGEO_SEGMENTS.CLIENT_ID))
                .map(convertibleProperty(HyperGeoSegment.COVERING_GEO, HYPERGEO_SEGMENTS.COVERING_GEO,
                        AdGroupMappings::geoFromDb, AdGroupMappings::geoToDb))
                .map(convertibleProperty(HyperGeoSegment.SEGMENT_DETAILS, HYPERGEO_SEGMENTS.GEOSEGMENT_DETAILS,
                        HyperGeoSegmentMappings::geoSegmentDetailsFromJson,
                        HyperGeoSegmentMappings::geoSegmentDetailsToJson))
                .build();
    }

    public Map<Long, HyperGeoSegment> getHyperGeoSegmentById(int shard, ClientId clientId,
                                                             Collection<Long> hyperGeoSegmentIds) {
        var condition = HYPERGEO_SEGMENTS.CLIENT_ID.eq(clientId.asLong());
        return getHyperGeoSegmentById(shard, condition, hyperGeoSegmentIds);
    }

    public Map<Long, HyperGeoSegment> getHyperGeoSegmentById(int shard, Condition condition,
                                                             Collection<Long> hyperGeoSegmentIds) {
        if (hyperGeoSegmentIds.isEmpty()) {
            return emptyMap();
        }

        return dslContextProvider.ppc(shard)
                .select(hyperGeoSegmentMapper.getFieldsToRead())
                .from(HYPERGEO_SEGMENTS)
                .where(HYPERGEO_SEGMENTS.GOAL_ID.in(hyperGeoSegmentIds))
                .and(condition)
                .fetchMap(HYPERGEO_SEGMENTS.GOAL_ID, hyperGeoSegmentMapper::fromDb);
    }

    /**
     * Возвращает те id сегментов гипер гео, для которых не существует самих гипер гео
     * (т.е. записей в retargeting_goals и, соответственно, в retargeting_conditions).
     *
     * @param shard              - номер шарда
     * @param hyperGeoSegmentIds - id сегментов гипер гео
     * @return Список id сегментов
     */
    public Set<Long> getUnusedHyperGeoSegmentIds(int shard,
                                                 Collection<Long> hyperGeoSegmentIds) {
        return getUnusedHyperGeoSegmentIds(dslContextProvider.ppc(shard), hyperGeoSegmentIds);
    }

    /**
     * Возвращает те id сегментов гипер гео, для которых не существует самих гипер гео
     * (т.е. записей в retargeting_goals и, соответственно, в retargeting_conditions).
     *
     * @param dslContext         - контекст транзакции
     * @param hyperGeoSegmentIds - id сегментов гипер гео
     * @return Список id сегментов
     */
    public Set<Long> getUnusedHyperGeoSegmentIds(DSLContext dslContext,
                                                 Collection<Long> hyperGeoSegmentIds) {
        if (isEmpty(hyperGeoSegmentIds)) {
            return Set.of();
        }

        Condition condition = HYPERGEO_SEGMENTS.GOAL_ID.in(hyperGeoSegmentIds);
        return getUnusedHyperGeoSegmentIds(dslContext, condition, null);
    }

    /**
     * ИСПОЛЬЗУЕТСЯ ТОЛЬКО В ДЖОБЕ И ВАНШОТЕ DeleteUnusedHyperGeos*.
     * Тесты на метод смотреть в DeleteUnusedHyperGeosOneshotTest.
     * <p>
     * Пример запроса - https://yql.yandex-team.ru/Operations/Yk2j9wVK8Jv0aYqEBDissg_IEZ2nTNdxkRXjoa0C8TA=
     */
    @QueryWithoutIndex("Получение id неиспользуемых сегментов гипер гео для их дальнейшего удаления;" +
            " редкая операция - используется в ежедневной джобе и в ваншоте")
    public Set<Long> getUnusedHyperGeoSegmentIds(int shard, Set<Long> idsToSkip, int limit) {
        var condition = isEmpty(idsToSkip)
                ? DSL.noCondition()
                : HYPERGEO_SEGMENTS.GOAL_ID.notIn(idsToSkip);
        return getUnusedHyperGeoSegmentIds(dslContextProvider.ppc(shard), condition, limit);
    }

    private Set<Long> getUnusedHyperGeoSegmentIds(DSLContext dslContext,
                                                  Condition condition,
                                                  @Nullable Integer limit) {
        condition = condition
                .and(RETARGETING_GOALS.GOAL_ID.isNull());

        var selectConditionStep = dslContext
                .select(HYPERGEO_SEGMENTS.GOAL_ID)
                .from(HYPERGEO_SEGMENTS)
                .leftJoin(RETARGETING_GOALS)
                .on(HYPERGEO_SEGMENTS.GOAL_ID.eq(RETARGETING_GOALS.GOAL_ID))
                .where(condition);

        if (limit != null) {
            selectConditionStep
                    .limit(limit);
        }

        return selectConditionStep
                .fetchSet(HYPERGEO_SEGMENTS.GOAL_ID);
    }

    public Set<Long> getHyperGeoSegmentIdsByHyperGeoId(int shard, @Nullable ClientId clientId,
                                                       Collection<Long> hyperGeoIds) {
        return getHyperGeoSegmentIdsByHyperGeoId(dslContextProvider.ppc(shard), clientId, hyperGeoIds);
    }

    public Set<Long> getHyperGeoSegmentIdsByHyperGeoId(DSLContext dslContext, @Nullable ClientId clientId,
                                                       Collection<Long> hyperGeoIds) {
        if (hyperGeoIds.isEmpty()) {
            return Set.of();
        }

        Condition condition = RETARGETING_CONDITIONS.RET_COND_ID.in(hyperGeoIds)
                .and(RETARGETING_CONDITIONS.RETARGETING_CONDITIONS_TYPE.eq(geo_segments));
        if (clientId != null) {
            condition = condition.and(RETARGETING_CONDITIONS.CLIENT_ID.eq(clientId.asLong()));
        }

        return dslContext
                .select(HYPERGEO_SEGMENTS.GOAL_ID)
                .from(HYPERGEO_SEGMENTS)
                .join(RETARGETING_GOALS).on(HYPERGEO_SEGMENTS.GOAL_ID.eq(RETARGETING_GOALS.GOAL_ID))
                .join(RETARGETING_CONDITIONS).on(RETARGETING_GOALS.RET_COND_ID.eq(RETARGETING_CONDITIONS.RET_COND_ID))
                .where(condition)
                .fetchSet(HYPERGEO_SEGMENTS.GOAL_ID);
    }

    public void addHyperGeoSegments(int shard, Collection<HyperGeoSegment> hyperGeoSegments) {
        if (hyperGeoSegments.isEmpty()) {
            return;
        }

        InsertHelper<HypergeoSegmentsRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), HYPERGEO_SEGMENTS);

        insertHelper.addAll(hyperGeoSegmentMapper, hyperGeoSegments);

        insertHelper.execute();
    }

    public void deleteHyperGeoSegmentById(int shard, @Nullable Collection<Long> hyperGeoSegmentIds) {
        deleteHyperGeoSegmentById(dslContextProvider.ppc(shard), hyperGeoSegmentIds);
    }

    public void deleteHyperGeoSegmentById(DSLContext dslContext, @Nullable Collection<Long> hyperGeoSegmentIds) {
        if (isEmpty(hyperGeoSegmentIds)) {
            return;
        }

        logger.trace("Try to delete hypergeo segments with following ids: {}", hyperGeoSegmentIds);
        dslContext
                .deleteFrom(HYPERGEO_SEGMENTS)
                .where(HYPERGEO_SEGMENTS.GOAL_ID.in(hyperGeoSegmentIds))
                .execute();
    }

    /**
     * ИСПОЛЬЗУЕТСЯ ТОЛЬКО ДЛЯ ЛОГИРОВАНИЯ В ДЖОБЕ И ВАНШОТЕ DeleteUnusedHyperGeos*.
     */
    public Collection<HyperGeoSegment> getHyperGeoSegmentsToLog(int shard, Collection<Long> hyperGeoSegmentIds) {
        return getHyperGeoSegmentById(shard, DSL.noCondition(), hyperGeoSegmentIds).values();
    }
}
