package ru.yandex.direct.core.entity.hypergeo.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.jooq.Configuration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityService;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeo;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeoSegment;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeoSegmentDetails;
import ru.yandex.direct.core.entity.hypergeo.model.HyperGeoSimple;
import ru.yandex.direct.core.entity.hypergeo.operation.HyperGeoOperationsFactory;
import ru.yandex.direct.core.entity.hypergeo.operation.HyperGeoSegmentsAddOperation;
import ru.yandex.direct.core.entity.hypergeo.operation.HyperGeoSegmentsDeleteOperation;
import ru.yandex.direct.core.entity.hypergeo.operation.HyperGeosAddOperation;
import ru.yandex.direct.core.entity.hypergeo.operation.HyperGeosDeleteOperation;
import ru.yandex.direct.core.entity.hypergeo.repository.HyperGeoRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.operation.Applicability.FULL;
import static ru.yandex.direct.operation.Applicability.PARTIAL;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class HyperGeoService implements EntityService<HyperGeoSimple, Long> {

    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;
    private final HyperGeoRepository hyperGeoRepository;
    private final HyperGeoOperationsFactory hyperGeoOperationsFactory;

    @Autowired
    public HyperGeoService(ShardHelper shardHelper,
                           DslContextProvider dslContextProvider,
                           HyperGeoRepository hyperGeoRepository,
                           HyperGeoOperationsFactory hyperGeoOperationsFactory) {
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
        this.hyperGeoRepository = hyperGeoRepository;
        this.hyperGeoOperationsFactory = hyperGeoOperationsFactory;
    }

    @Override
    public List<HyperGeoSimple> get(ClientId clientId, Long operatorUid, Collection<Long> hyperGeoIds) {
        return EntryStream.of(getHyperGeoSimpleById(clientId, hyperGeoIds))
                .values()
                .toList();
    }

    @Override
    public MassResult<Long> add(ClientId clientId,
                                Long operatorUid,
                                List<HyperGeoSimple> hyperGeos,
                                Applicability applicability) {
        return createHyperGeos(clientId, hyperGeos);
    }

    @Override
    public MassResult<Long> copy(CopyOperationContainer copyContainer,
                                 List<HyperGeoSimple> hyperGeos,
                                 Applicability applicability) {
        checkState(!copyContainer.isCopyingBetweenClients());
        return createHyperGeos(copyContainer.getShardTo(), copyContainer.getClientIdTo(), hyperGeos);
    }

    public CreateHyperGeoResultContainer createHyperGeoSegments(ClientId clientId, String login,
                                                                Collection<HyperGeoSegmentDetails> hyperGeoSegmentsDetails) {
        int shard = shardHelper.getShardByClientId(clientId);

        List<HyperGeoSegment> models = mapList(hyperGeoSegmentsDetails, hyperGeoSegmentDetails ->
                new HyperGeoSegment()
                        .withClientId(clientId.asLong())
                        .withSegmentDetails(hyperGeoSegmentDetails));

        HyperGeoSegmentsAddOperation addOperation =
                hyperGeoOperationsFactory.createHyperGeoSegmentsAddOperation(FULL, models, shard, login);

        var result = addOperation.prepareAndApply();

        return new CreateHyperGeoResultContainer(result, models);
    }

    public MassResult<Long> createHyperGeos(ClientId clientId, List<HyperGeoSimple> hyperGeos) {
        int shard = shardHelper.getShardByClientId(clientId);
        return createHyperGeos(shard, clientId, hyperGeos);
    }

    private MassResult<Long> createHyperGeos(int shard, ClientId clientId, List<HyperGeoSimple> hyperGeos) {
        HyperGeosAddOperation addOperation =
                hyperGeoOperationsFactory.createHyperGeosAddOperation(FULL, hyperGeos, shard, clientId);

        return addOperation.prepareAndApply();
    }


    public Map<Long, HyperGeo> getHyperGeoByAdGroupId(ClientId clientId, Collection<Long> adGroupIds) {
        int shard = shardHelper.getShardByClientId(clientId);

        return hyperGeoRepository.getHyperGeoByAdGroupId(shard, clientId, adGroupIds);
    }

    public HyperGeo getHyperGeoById(ClientId clientId, Long hyperGeoId) {
        return getHyperGeoById(clientId, List.of(hyperGeoId)).get(hyperGeoId);
    }

    public Map<Long, HyperGeo> getHyperGeoById(ClientId clientId, Collection<Long> hyperGeoIds) {
        int shard = shardHelper.getShardByClientId(clientId);

        return hyperGeoRepository.getHyperGeoById(shard, clientId, hyperGeoIds);
    }

    public Map<Long, HyperGeoSimple> getHyperGeoSimpleById(ClientId clientId, Collection<Long> hyperGeoIds) {
        int shard = shardHelper.getShardByClientId(clientId);

        return hyperGeoRepository.getHyperGeoSimpleById(shard, clientId, hyperGeoIds);
    }

    public Map<Long, List<Long>> getAdGroupIdsByHyperGeoId(ClientId clientId, Collection<Long> hyperGeoIds) {
        int shard = shardHelper.getShardByClientId(clientId);

        return hyperGeoRepository.getAdGroupIdsByHyperGeoId(shard, hyperGeoIds);
    }

    /**
     * Удаляет неиспользуемые гипер гео и их сегменты.
     * Неиспользуемыми считаются те гипер гео, чьи id (ret_cond_id) не привязаны ни к какой группе в таблице
     * adgroups_hypergeo_retargetings, и чьи гео сегменты (goal_id) не используются в adgroup_additional_targetings.
     *
     * @param shard       - номер шарда
     * @param clientId    - id клиента
     * @param hyperGeoIds - id гипер гео, которые нужно удалить
     *                    (ret_cond_id в retargeting_conditions с типом geo_segments)
     */
    public void deleteHyperGeos(int shard, @Nullable ClientId clientId, List<Long> hyperGeoIds) {
        var context = dslContextProvider.ppc(shard);
        HyperGeosDeleteOperation deleteOperation =
                hyperGeoOperationsFactory.createHyperGeosDeleteOperation(PARTIAL, hyperGeoIds, context, clientId);

        deleteOperation.prepareAndApply();
    }

    /**
     * Удаляет неиспользуемые гипер гео и их сегменты.
     * Неиспользуемыми считаются те гипер гео, чьи id (ret_cond_id) не привязаны ни к какой группе в таблице
     * adgroups_hypergeo_retargetings, и чьи гео сегменты (goal_id) не используются в adgroup_additional_targetings.
     *
     * @param config      - конфигурация
     * @param clientId    - id клиента
     * @param hyperGeoIds - id гипер гео, которые нужно удалить
     *                    (ret_cond_id в retargeting_conditions с типом geo_segments)
     */
    public void deleteHyperGeos(Configuration config, @Nullable ClientId clientId, List<Long> hyperGeoIds) {
        HyperGeosDeleteOperation deleteOperation =
                hyperGeoOperationsFactory.createHyperGeosDeleteOperation(PARTIAL, hyperGeoIds, config.dsl(), clientId);

        deleteOperation.prepareAndApply();
    }

    /**
     * Удаляет только неиспользуемые сегменты гипер гео.
     * Удалить можно те сегменты гипер гео, которые не привязаны ни к одному гипер гео (retargeting_conditions),
     * т.е. не имеют записей в retargeting_goals со своим goal_id. Если нужно удалить привязанный к гипер гео сегмент,
     * стоит воспользоваться методом deleteHyperGeos.
     *
     * @param shard              - номер шарда
     * @param hyperGeoSegmentIds - id сегментов гипер гео, которые нужно удалить (goal_id в retargeting_goals)
     */
    public void deleteHyperGeoSegments(int shard, List<Long> hyperGeoSegmentIds) {
        HyperGeoSegmentsDeleteOperation deleteOperation =
                hyperGeoOperationsFactory.createHyperGeoSegmentsDeleteOperation(PARTIAL, hyperGeoSegmentIds, shard);

        deleteOperation.prepareAndApply();
    }
}
